<?php
/**
 * Plugin Name: YourBeans Coffee
 * Plugin URI: https://myshop.coffee/yourbeans
 * Description: Premium coffee dropshipping integration - Import and sell coffee products with automated fulfillment from MyShop.Coffee
 * Version: 1.0.0
 * Author: MyShop Enterprises Pty Ltd
 * Author URI: https://myshop.coffee
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: yourbeans-coffee
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 6.0
 * WC tested up to: 8.4
 *
 * @package YourBeans_Coffee
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('YOURBEANS_VERSION', '1.0.0');
define('YOURBEANS_PLUGIN_FILE', __FILE__);
define('YOURBEANS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('YOURBEANS_PLUGIN_URL', plugin_dir_url(__FILE__));
define('YOURBEANS_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Check if WooCommerce is active
 */
function yourbeans_check_woocommerce() {
    if (!class_exists('WooCommerce')) {
        add_action('admin_notices', 'yourbeans_woocommerce_missing_notice');
        return false;
    }
    return true;
}

/**
 * WooCommerce missing notice
 */
function yourbeans_woocommerce_missing_notice() {
    ?>
    <div class="notice notice-error">
        <p><?php esc_html_e('YourBeans Coffee requires WooCommerce to be installed and active.', 'yourbeans-coffee'); ?></p>
    </div>
    <?php
}

/**
 * Initialize the plugin
 */
function yourbeans_init() {
    // Check WooCommerce dependency
    if (!yourbeans_check_woocommerce()) {
        return;
    }

    // Load text domain
    load_plugin_textdomain('yourbeans-coffee', false, dirname(YOURBEANS_PLUGIN_BASENAME) . '/languages');

    // Include required files
    require_once YOURBEANS_PLUGIN_DIR . 'includes/class-yourbeans-core.php';

    // Initialize the plugin
    YourBeans_Core::instance();
}
add_action('plugins_loaded', 'yourbeans_init', 20);

/**
 * Activation hook
 */
function yourbeans_activate() {
    // Check PHP version
    if (version_compare(PHP_VERSION, '7.4', '<')) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(
            esc_html__('YourBeans Coffee requires PHP 7.4 or higher.', 'yourbeans-coffee'),
            'Plugin Activation Error',
            array('back_link' => true)
        );
    }

    // Check WooCommerce
    if (!class_exists('WooCommerce')) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(
            esc_html__('YourBeans Coffee requires WooCommerce to be installed and active.', 'yourbeans-coffee'),
            'Plugin Activation Error',
            array('back_link' => true)
        );
    }

    // Set default options
    if (!get_option('yourbeans_default_markup')) {
        add_option('yourbeans_default_markup', 50);
    }

    // Flush rewrite rules
    flush_rewrite_rules();

    // Log activation
    error_log('YourBeans Coffee plugin activated - Version ' . YOURBEANS_VERSION);
}
register_activation_hook(__FILE__, 'yourbeans_activate');

/**
 * Deactivation hook
 */
function yourbeans_deactivate() {
    // Flush rewrite rules
    flush_rewrite_rules();

    // Log deactivation
    error_log('YourBeans Coffee plugin deactivated');
}
register_deactivation_hook(__FILE__, 'yourbeans_deactivate');

/**
 * Declare HPOS compatibility
 */
add_action('before_woocommerce_init', function() {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});
