<?php
/**
 * YourBeans Customer Subscriptions Class
 *
 * Built-in subscription functionality for store owners to sell coffee subscriptions
 * to their customers (weekly, fortnightly, monthly deliveries).
 *
 * @package YourBeans_Coffee
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class YourBeans_Subscriptions
 */
class YourBeans_Subscriptions {

    /**
     * Subscription intervals
     *
     * @var array
     */
    private $intervals = array(
        'weekly' => array(
            'label' => 'Weekly',
            'days' => 7,
        ),
        'fortnightly' => array(
            'label' => 'Every 2 Weeks',
            'days' => 14,
        ),
        'monthly' => array(
            'label' => 'Monthly',
            'days' => 30,
        ),
    );

    /**
     * Constructor
     */
    public function __construct() {
        // Frontend subscription options
        add_action('woocommerce_before_add_to_cart_button', array($this, 'display_subscription_options'));
        add_filter('woocommerce_add_cart_item_data', array($this, 'add_subscription_to_cart'), 10, 3);
        add_filter('woocommerce_get_item_data', array($this, 'display_subscription_in_cart'), 10, 2);
        add_action('woocommerce_before_calculate_totals', array($this, 'apply_subscription_discount'));

        // Order processing
        add_action('woocommerce_checkout_create_order_line_item', array($this, 'save_subscription_to_order'), 10, 4);
        add_action('woocommerce_order_status_completed', array($this, 'schedule_renewal'));

        // Subscription renewals
        add_action('yourbeans_process_subscription_renewal', array($this, 'process_renewal'), 10, 2);

        // Customer subscription management
        add_action('init', array($this, 'register_subscription_endpoint'));
        add_filter('woocommerce_account_menu_items', array($this, 'add_subscription_menu_item'));
        add_action('woocommerce_account_yourbeans-subscriptions_endpoint', array($this, 'subscription_management_page'));

        // AJAX handlers
        add_action('wp_ajax_yourbeans_pause_subscription', array($this, 'ajax_pause_subscription'));
        add_action('wp_ajax_yourbeans_cancel_subscription', array($this, 'ajax_cancel_subscription'));
        add_action('wp_ajax_yourbeans_resume_subscription', array($this, 'ajax_resume_subscription'));

        // Frontend scripts
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_scripts'));
    }

    /**
     * Display subscription options on product page
     */
    public function display_subscription_options() {
        global $product;

        if (!$this->is_yourbeans_product($product)) {
            return;
        }

        $subscription_enabled = get_post_meta($product->get_id(), '_yourbeans_subscription_enabled', true);

        if ($subscription_enabled !== 'yes') {
            return;
        }

        $discount = intval(get_post_meta($product->get_id(), '_yourbeans_subscription_discount', true));
        if ($discount <= 0) {
            $discount = intval(get_option('yourbeans_default_subscription_discount', 10));
        }
        ?>
        <div class="yourbeans-subscription-options">
            <h4><?php esc_html_e('Purchase Options', 'yourbeans-coffee'); ?></h4>

            <label class="subscription-option">
                <input type="radio" name="yourbeans_purchase_type" value="one-time" checked>
                <span class="option-label"><?php esc_html_e('One-time purchase', 'yourbeans-coffee'); ?></span>
            </label>

            <label class="subscription-option subscription-choice">
                <input type="radio" name="yourbeans_purchase_type" value="subscription">
                <span class="option-label">
                    <?php
                    printf(
                        esc_html__('Subscribe & Save %d%%', 'yourbeans-coffee'),
                        $discount
                    );
                    ?>
                </span>
                <span class="option-description">
                    <?php esc_html_e('Delivered automatically, cancel anytime', 'yourbeans-coffee'); ?>
                </span>
            </label>

            <div class="subscription-interval" style="display:none;">
                <label for="yourbeans_interval"><?php esc_html_e('Delivery Frequency:', 'yourbeans-coffee'); ?></label>
                <select name="yourbeans_interval" id="yourbeans_interval">
                    <?php foreach ($this->intervals as $key => $interval) : ?>
                        <option value="<?php echo esc_attr($key); ?>"<?php selected($key, 'fortnightly'); ?>>
                            <?php echo esc_html($interval['label']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
        </div>

        <style>
            .yourbeans-subscription-options {
                margin: 20px 0;
                padding: 15px;
                background: #f8f8f8;
                border-radius: 4px;
            }
            .yourbeans-subscription-options h4 {
                margin: 0 0 15px 0;
                font-size: 16px;
            }
            .subscription-option {
                display: block;
                padding: 12px;
                margin-bottom: 8px;
                background: #fff;
                border: 2px solid #ddd;
                border-radius: 4px;
                cursor: pointer;
                transition: border-color 0.2s;
            }
            .subscription-option:hover,
            .subscription-option.selected {
                border-color: #0073aa;
            }
            .subscription-option input[type="radio"] {
                margin-right: 10px;
            }
            .subscription-option .option-label {
                font-weight: 600;
            }
            .subscription-option .option-description {
                display: block;
                margin-left: 25px;
                font-size: 12px;
                color: #666;
            }
            .subscription-interval {
                margin-top: 15px;
                padding: 15px;
                background: #e8f4fc;
                border-radius: 4px;
            }
            .subscription-interval label {
                display: block;
                margin-bottom: 8px;
                font-weight: 600;
            }
            .subscription-interval select {
                width: 100%;
                padding: 8px;
            }
        </style>

        <script>
        jQuery(function($) {
            var $options = $('input[name="yourbeans_purchase_type"]');
            var $intervalContainer = $('.subscription-interval');

            $options.on('change', function() {
                $('.subscription-option').removeClass('selected');
                $(this).closest('.subscription-option').addClass('selected');

                if ($(this).val() === 'subscription') {
                    $intervalContainer.slideDown();
                } else {
                    $intervalContainer.slideUp();
                }
            });

            // Initialize
            $options.filter(':checked').closest('.subscription-option').addClass('selected');
        });
        </script>
        <?php
    }

    /**
     * Add subscription data to cart
     *
     * @param array $cart_item_data Cart item data
     * @param int   $product_id Product ID
     * @param int   $variation_id Variation ID
     * @return array Modified cart item data
     */
    public function add_subscription_to_cart($cart_item_data, $product_id, $variation_id) {
        if (isset($_POST['yourbeans_purchase_type']) && $_POST['yourbeans_purchase_type'] === 'subscription') {
            $interval = isset($_POST['yourbeans_interval']) ? sanitize_text_field($_POST['yourbeans_interval']) : 'monthly';

            // Validate interval
            if (!isset($this->intervals[$interval])) {
                $interval = 'monthly';
            }

            // Get discount
            $discount = intval(get_post_meta($product_id, '_yourbeans_subscription_discount', true));
            if ($discount <= 0) {
                $discount = intval(get_option('yourbeans_default_subscription_discount', 10));
            }

            $cart_item_data['yourbeans_subscription'] = array(
                'type' => 'subscription',
                'interval' => $interval,
                'discount' => $discount,
            );

            // Generate unique key to allow multiple subscriptions in cart
            $cart_item_data['unique_key'] = md5(microtime() . rand());
        }

        return $cart_item_data;
    }

    /**
     * Display subscription info in cart
     *
     * @param array $item_data Item display data
     * @param array $cart_item Cart item
     * @return array Modified item data
     */
    public function display_subscription_in_cart($item_data, $cart_item) {
        if (isset($cart_item['yourbeans_subscription'])) {
            $subscription = $cart_item['yourbeans_subscription'];
            $interval_label = isset($this->intervals[$subscription['interval']]['label'])
                ? $this->intervals[$subscription['interval']]['label']
                : $subscription['interval'];

            $item_data[] = array(
                'key' => __('Subscription', 'yourbeans-coffee'),
                'value' => sprintf(
                    '%s (%d%% off)',
                    $interval_label,
                    $subscription['discount']
                ),
            );
        }

        return $item_data;
    }

    /**
     * Apply subscription discount to cart items
     *
     * @param WC_Cart $cart Cart object
     */
    public function apply_subscription_discount($cart) {
        if (is_admin() && !defined('DOING_AJAX')) {
            return;
        }

        if (did_action('woocommerce_before_calculate_totals') >= 2) {
            return;
        }

        foreach ($cart->get_cart() as $cart_item) {
            if (isset($cart_item['yourbeans_subscription'])) {
                $discount = floatval($cart_item['yourbeans_subscription']['discount']) / 100;
                $original_price = floatval($cart_item['data']->get_regular_price());
                $discounted_price = $original_price * (1 - $discount);

                $cart_item['data']->set_price($discounted_price);
            }
        }
    }

    /**
     * Save subscription data to order
     *
     * @param WC_Order_Item_Product $item Order item
     * @param string                $cart_item_key Cart item key
     * @param array                 $values Cart item values
     * @param WC_Order              $order Order object
     */
    public function save_subscription_to_order($item, $cart_item_key, $values, $order) {
        if (isset($values['yourbeans_subscription'])) {
            $item->add_meta_data('_yourbeans_subscription', $values['yourbeans_subscription'], true);
        }
    }

    /**
     * Schedule next renewal after order completion
     *
     * @param int $order_id Order ID
     */
    public function schedule_renewal($order_id) {
        $order = wc_get_order($order_id);

        if (!$order) {
            return;
        }

        foreach ($order->get_items() as $item) {
            $subscription_data = $item->get_meta('_yourbeans_subscription');

            if (!$subscription_data || !isset($subscription_data['type']) || $subscription_data['type'] !== 'subscription') {
                continue;
            }

            $interval = $subscription_data['interval'];
            $days = isset($this->intervals[$interval]['days']) ? $this->intervals[$interval]['days'] : 30;
            $next_renewal = strtotime('+' . $days . ' days');

            // Create subscription record
            $subscription_id = $this->create_subscription_record($order, $item, $subscription_data);

            if ($subscription_id) {
                // Schedule renewal
                wp_schedule_single_event($next_renewal, 'yourbeans_process_subscription_renewal', array(
                    $subscription_id,
                    $order->get_customer_id(),
                ));

                error_log(sprintf(
                    'YourBeans: Scheduled subscription renewal #%d for customer #%d on %s',
                    $subscription_id,
                    $order->get_customer_id(),
                    date('Y-m-d H:i:s', $next_renewal)
                ));
            }
        }
    }

    /**
     * Create subscription record in database
     *
     * @param WC_Order              $order Original order
     * @param WC_Order_Item_Product $item Order item
     * @param array                 $subscription_data Subscription data
     * @return int|false Subscription ID or false on failure
     */
    private function create_subscription_record($order, $item, $subscription_data) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'yourbeans_subscriptions';

        // Ensure table exists
        $this->maybe_create_subscriptions_table();

        $product_id = $item->get_product_id();
        $variation_id = $item->get_variation_id();

        $data = array(
            'customer_id' => $order->get_customer_id(),
            'original_order_id' => $order->get_id(),
            'product_id' => $product_id,
            'variation_id' => $variation_id ? $variation_id : 0,
            'quantity' => $item->get_quantity(),
            'interval' => $subscription_data['interval'],
            'discount' => $subscription_data['discount'],
            'status' => 'active',
            'next_renewal' => date('Y-m-d H:i:s', strtotime('+' . $this->intervals[$subscription_data['interval']]['days'] . ' days')),
            'created_at' => current_time('mysql'),
        );

        $result = $wpdb->insert($table_name, $data, array(
            '%d', '%d', '%d', '%d', '%d', '%s', '%d', '%s', '%s', '%s'
        ));

        if ($result) {
            return $wpdb->insert_id;
        }

        error_log('YourBeans: Failed to create subscription record - ' . $wpdb->last_error);
        return false;
    }

    /**
     * Process subscription renewal
     *
     * @param int $subscription_id Subscription ID
     * @param int $customer_id Customer ID
     */
    public function process_renewal($subscription_id, $customer_id) {
        $subscription = $this->get_subscription($subscription_id);

        if (!$subscription || $subscription['status'] !== 'active') {
            error_log(sprintf('YourBeans: Skipping renewal for subscription #%d - status: %s',
                $subscription_id,
                $subscription ? $subscription['status'] : 'not found'
            ));
            return;
        }

        // Check if customer still exists
        $customer = get_user_by('id', $customer_id);
        if (!$customer) {
            $this->update_subscription_status($subscription_id, 'cancelled');
            error_log(sprintf('YourBeans: Cancelled subscription #%d - customer not found', $subscription_id));
            return;
        }

        // Create renewal order
        $renewal_order = $this->create_renewal_order($subscription, $customer_id);

        if (is_wp_error($renewal_order)) {
            error_log(sprintf('YourBeans: Failed to create renewal order for subscription #%d - %s',
                $subscription_id,
                $renewal_order->get_error_message()
            ));
            return;
        }

        // Update subscription
        $days = $this->intervals[$subscription['interval']]['days'];
        $next_renewal = strtotime('+' . $days . ' days');

        $this->update_subscription_next_renewal($subscription_id, date('Y-m-d H:i:s', $next_renewal));

        // Schedule next renewal
        wp_schedule_single_event($next_renewal, 'yourbeans_process_subscription_renewal', array(
            $subscription_id,
            $customer_id,
        ));

        // Send renewal notification
        $this->send_renewal_notification($subscription, $renewal_order);

        error_log(sprintf('YourBeans: Created renewal order #%d for subscription #%d',
            $renewal_order->get_id(),
            $subscription_id
        ));
    }

    /**
     * Create renewal order
     *
     * @param array $subscription Subscription data
     * @param int   $customer_id Customer ID
     * @return WC_Order|WP_Error Order object or error
     */
    private function create_renewal_order($subscription, $customer_id) {
        $original_order = wc_get_order($subscription['original_order_id']);

        if (!$original_order) {
            return new WP_Error('original_order_not_found', 'Original order not found');
        }

        // Create new order
        $renewal_order = wc_create_order(array(
            'customer_id' => $customer_id,
            'status' => 'pending',
        ));

        if (is_wp_error($renewal_order)) {
            return $renewal_order;
        }

        // Add product
        $product = wc_get_product($subscription['variation_id'] ? $subscription['variation_id'] : $subscription['product_id']);

        if (!$product) {
            return new WP_Error('product_not_found', 'Product not found');
        }

        // Calculate discounted price
        $original_price = floatval($product->get_regular_price());
        $discount = floatval($subscription['discount']) / 100;
        $discounted_price = $original_price * (1 - $discount);

        $renewal_order->add_product($product, $subscription['quantity'], array(
            'subtotal' => $discounted_price * $subscription['quantity'],
            'total' => $discounted_price * $subscription['quantity'],
        ));

        // Copy addresses
        $renewal_order->set_address($original_order->get_address('billing'), 'billing');
        $renewal_order->set_address($original_order->get_address('shipping'), 'shipping');

        // Add meta
        $renewal_order->add_meta_data('_yourbeans_subscription_id', $subscription['id']);
        $renewal_order->add_meta_data('_yourbeans_renewal_order', 'yes');

        // Calculate totals
        $renewal_order->calculate_totals();
        $renewal_order->save();

        // Attempt to charge saved payment method
        $this->charge_renewal($renewal_order, $customer_id);

        return $renewal_order;
    }

    /**
     * Charge renewal order using saved payment method
     *
     * @param WC_Order $order Order object
     * @param int      $customer_id Customer ID
     */
    private function charge_renewal($order, $customer_id) {
        // Get customer's saved payment token
        $tokens = WC_Payment_Tokens::get_customer_tokens($customer_id, 'stripe');

        if (empty($tokens)) {
            // No saved payment method - set order to pending payment
            $order->add_order_note(__('Awaiting payment - no saved payment method available.', 'yourbeans-coffee'));
            return;
        }

        // Get default token
        $default_token = WC_Payment_Tokens::get_customer_default_token($customer_id);

        if (!$default_token) {
            $default_token = reset($tokens);
        }

        // Attempt payment via WooCommerce Stripe
        if (class_exists('WC_Gateway_Stripe')) {
            try {
                $order->set_payment_method('stripe');
                $order->add_meta_data('_stripe_customer_id', get_user_meta($customer_id, '_stripe_customer_id', true));
                $order->add_meta_data('_stripe_source_id', $default_token->get_token());
                $order->save();

                // Trigger payment
                do_action('woocommerce_scheduled_subscription_payment_stripe', $order->get_total(), $order);

            } catch (Exception $e) {
                $order->add_order_note(sprintf(
                    __('Renewal payment failed: %s', 'yourbeans-coffee'),
                    $e->getMessage()
                ));
            }
        }
    }

    /**
     * Send renewal notification email
     *
     * @param array    $subscription Subscription data
     * @param WC_Order $order Renewal order
     */
    private function send_renewal_notification($subscription, $order) {
        $customer = get_user_by('id', $subscription['customer_id']);

        if (!$customer) {
            return;
        }

        $product = wc_get_product($subscription['variation_id'] ? $subscription['variation_id'] : $subscription['product_id']);
        $product_name = $product ? $product->get_name() : __('Coffee Subscription', 'yourbeans-coffee');

        $to = $customer->user_email;
        $subject = sprintf(__('Your %s subscription renewal', 'yourbeans-coffee'), get_bloginfo('name'));

        $message = sprintf(
            __('Hi %s,

Your coffee subscription has been renewed!

Product: %s
Order: #%d
Total: %s

View your order: %s

Thank you for your continued support!

%s', 'yourbeans-coffee'),
            $customer->first_name ? $customer->first_name : $customer->display_name,
            $product_name,
            $order->get_id(),
            $order->get_formatted_order_total(),
            $order->get_view_order_url(),
            get_bloginfo('name')
        );

        wp_mail($to, $subject, $message);
    }

    /**
     * Get subscription by ID
     *
     * @param int $subscription_id Subscription ID
     * @return array|null Subscription data or null
     */
    public function get_subscription($subscription_id) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'yourbeans_subscriptions';

        return $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $subscription_id),
            ARRAY_A
        );
    }

    /**
     * Get customer subscriptions
     *
     * @param int $customer_id Customer ID
     * @return array Subscriptions
     */
    public function get_customer_subscriptions($customer_id) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'yourbeans_subscriptions';

        return $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM $table_name WHERE customer_id = %d ORDER BY created_at DESC",
                $customer_id
            ),
            ARRAY_A
        );
    }

    /**
     * Update subscription status
     *
     * @param int    $subscription_id Subscription ID
     * @param string $status New status
     * @return bool Success
     */
    private function update_subscription_status($subscription_id, $status) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'yourbeans_subscriptions';

        return $wpdb->update(
            $table_name,
            array('status' => $status),
            array('id' => $subscription_id),
            array('%s'),
            array('%d')
        ) !== false;
    }

    /**
     * Update subscription next renewal date
     *
     * @param int    $subscription_id Subscription ID
     * @param string $next_renewal Next renewal date
     * @return bool Success
     */
    private function update_subscription_next_renewal($subscription_id, $next_renewal) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'yourbeans_subscriptions';

        return $wpdb->update(
            $table_name,
            array('next_renewal' => $next_renewal),
            array('id' => $subscription_id),
            array('%s'),
            array('%d')
        ) !== false;
    }

    /**
     * Check if product is YourBeans product
     *
     * @param WC_Product $product Product object
     * @return bool
     */
    private function is_yourbeans_product($product) {
        if (!$product) {
            return false;
        }
        return !empty(get_post_meta($product->get_id(), '_yourbeans_myshop_product_id', true));
    }

    /**
     * Register subscription endpoint for My Account
     */
    public function register_subscription_endpoint() {
        add_rewrite_endpoint('yourbeans-subscriptions', EP_ROOT | EP_PAGES);
    }

    /**
     * Add subscription menu item to My Account
     *
     * @param array $items Menu items
     * @return array Modified items
     */
    public function add_subscription_menu_item($items) {
        $new_items = array();

        foreach ($items as $key => $value) {
            $new_items[$key] = $value;

            // Add after orders
            if ($key === 'orders') {
                $new_items['yourbeans-subscriptions'] = __('Coffee Subscriptions', 'yourbeans-coffee');
            }
        }

        return $new_items;
    }

    /**
     * Display subscription management page
     */
    public function subscription_management_page() {
        $customer_id = get_current_user_id();
        $subscriptions = $this->get_customer_subscriptions($customer_id);

        include YOURBEANS_PLUGIN_DIR . 'templates/myaccount/subscriptions.php';
    }

    /**
     * AJAX: Pause subscription
     */
    public function ajax_pause_subscription() {
        check_ajax_referer('yourbeans_subscription_nonce', 'nonce');

        $subscription_id = isset($_POST['subscription_id']) ? intval($_POST['subscription_id']) : 0;
        $subscription = $this->get_subscription($subscription_id);

        if (!$subscription || $subscription['customer_id'] != get_current_user_id()) {
            wp_send_json_error(array('message' => __('Invalid subscription.', 'yourbeans-coffee')));
        }

        // Remove scheduled renewal
        $timestamp = wp_next_scheduled('yourbeans_process_subscription_renewal', array(
            $subscription_id,
            $subscription['customer_id'],
        ));

        if ($timestamp) {
            wp_unschedule_event($timestamp, 'yourbeans_process_subscription_renewal', array(
                $subscription_id,
                $subscription['customer_id'],
            ));
        }

        $this->update_subscription_status($subscription_id, 'paused');

        wp_send_json_success(array('message' => __('Subscription paused.', 'yourbeans-coffee')));
    }

    /**
     * AJAX: Resume subscription
     */
    public function ajax_resume_subscription() {
        check_ajax_referer('yourbeans_subscription_nonce', 'nonce');

        $subscription_id = isset($_POST['subscription_id']) ? intval($_POST['subscription_id']) : 0;
        $subscription = $this->get_subscription($subscription_id);

        if (!$subscription || $subscription['customer_id'] != get_current_user_id()) {
            wp_send_json_error(array('message' => __('Invalid subscription.', 'yourbeans-coffee')));
        }

        // Schedule next renewal
        $days = $this->intervals[$subscription['interval']]['days'];
        $next_renewal = strtotime('+' . $days . ' days');

        wp_schedule_single_event($next_renewal, 'yourbeans_process_subscription_renewal', array(
            $subscription_id,
            $subscription['customer_id'],
        ));

        $this->update_subscription_status($subscription_id, 'active');
        $this->update_subscription_next_renewal($subscription_id, date('Y-m-d H:i:s', $next_renewal));

        wp_send_json_success(array('message' => __('Subscription resumed.', 'yourbeans-coffee')));
    }

    /**
     * AJAX: Cancel subscription
     */
    public function ajax_cancel_subscription() {
        check_ajax_referer('yourbeans_subscription_nonce', 'nonce');

        $subscription_id = isset($_POST['subscription_id']) ? intval($_POST['subscription_id']) : 0;
        $subscription = $this->get_subscription($subscription_id);

        if (!$subscription || $subscription['customer_id'] != get_current_user_id()) {
            wp_send_json_error(array('message' => __('Invalid subscription.', 'yourbeans-coffee')));
        }

        // Remove scheduled renewal
        $timestamp = wp_next_scheduled('yourbeans_process_subscription_renewal', array(
            $subscription_id,
            $subscription['customer_id'],
        ));

        if ($timestamp) {
            wp_unschedule_event($timestamp, 'yourbeans_process_subscription_renewal', array(
                $subscription_id,
                $subscription['customer_id'],
            ));
        }

        $this->update_subscription_status($subscription_id, 'cancelled');

        wp_send_json_success(array('message' => __('Subscription cancelled.', 'yourbeans-coffee')));
    }

    /**
     * Enqueue frontend scripts
     */
    public function enqueue_frontend_scripts() {
        if (is_account_page()) {
            wp_enqueue_script(
                'yourbeans-subscriptions',
                YOURBEANS_PLUGIN_URL . 'assets/js/subscriptions.js',
                array('jquery'),
                YOURBEANS_VERSION,
                true
            );

            wp_localize_script('yourbeans-subscriptions', 'yourbeansSubscriptions', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('yourbeans_subscription_nonce'),
                'strings' => array(
                    'confirm_pause' => __('Are you sure you want to pause this subscription?', 'yourbeans-coffee'),
                    'confirm_cancel' => __('Are you sure you want to cancel this subscription? This cannot be undone.', 'yourbeans-coffee'),
                ),
            ));
        }
    }

    /**
     * Create subscriptions table if it does not exist
     */
    private function maybe_create_subscriptions_table() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'yourbeans_subscriptions';

        if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") === $table_name) {
            return;
        }

        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $table_name (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            customer_id bigint(20) unsigned NOT NULL,
            original_order_id bigint(20) unsigned NOT NULL,
            product_id bigint(20) unsigned NOT NULL,
            variation_id bigint(20) unsigned DEFAULT 0,
            quantity int(11) NOT NULL DEFAULT 1,
            interval varchar(20) NOT NULL DEFAULT 'monthly',
            discount int(11) NOT NULL DEFAULT 10,
            status varchar(20) NOT NULL DEFAULT 'active',
            next_renewal datetime DEFAULT NULL,
            created_at datetime NOT NULL,
            PRIMARY KEY (id),
            KEY customer_id (customer_id),
            KEY status (status),
            KEY next_renewal (next_renewal)
        ) $charset_collate;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
    }

    /**
     * Get subscription intervals
     *
     * @return array Intervals
     */
    public function get_intervals() {
        return $this->intervals;
    }
}
