<?php
/**
 * YourBeans Stripe Connect Integration
 *
 * Handles Stripe Connect for automatic payment splitting
 *
 * @package YourBeans_Coffee
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class YourBeans_Stripe_Connect
 */
class YourBeans_Stripe_Connect {

    /**
     * MyShop's Stripe Connect Client ID
     *
     * @var string
     */
    private $client_id;

    /**
     * Constructor
     */
    public function __construct() {
        // MyShop's Stripe Connect platform credentials
        // These would come from MyShop.Coffee's Stripe dashboard
        $this->client_id = defined('YOURBEANS_STRIPE_CLIENT_ID') ? YOURBEANS_STRIPE_CLIENT_ID : '';

        // Handle OAuth callback
        add_action('init', array($this, 'handle_oauth_callback'));

        // Hook into WooCommerce Stripe payment
        add_filter('wc_stripe_payment_intent_args', array($this, 'add_application_fee'), 10, 2);
        add_filter('wc_stripe_generate_payment_request', array($this, 'modify_stripe_request'), 10, 3);

        // AJAX handlers
        add_action('wp_ajax_yourbeans_disconnect_stripe', array($this, 'ajax_disconnect_stripe'));
    }

    /**
     * Get Stripe Connect OAuth URL
     *
     * @return string OAuth URL
     */
    public function get_connect_url() {
        if (empty($this->client_id)) {
            return '';
        }

        $redirect_uri = admin_url('admin.php?page=yourbeans-settings&yourbeans_stripe_callback=1');

        $params = array(
            'response_type' => 'code',
            'client_id' => $this->client_id,
            'scope' => 'read_write',
            'redirect_uri' => $redirect_uri,
            'state' => wp_create_nonce('yourbeans_stripe_connect'),
            'stripe_user[business_type]' => 'company',
            'stripe_user[country]' => wc_get_base_location()['country'] ?? 'AU',
        );

        return 'https://connect.stripe.com/oauth/authorize?' . http_build_query($params);
    }

    /**
     * Handle Stripe OAuth callback
     */
    public function handle_oauth_callback() {
        if (!isset($_GET['yourbeans_stripe_callback'])) {
            return;
        }

        // Check for error
        if (isset($_GET['error'])) {
            add_settings_error(
                'yourbeans_messages',
                'stripe_error',
                sprintf(
                    __('Stripe connection failed: %s', 'yourbeans-coffee'),
                    sanitize_text_field($_GET['error_description'] ?? $_GET['error'])
                ),
                'error'
            );
            wp_redirect(admin_url('admin.php?page=yourbeans-settings'));
            exit;
        }

        // Check for authorization code
        if (!isset($_GET['code']) || !isset($_GET['state'])) {
            return;
        }

        // Verify nonce
        if (!wp_verify_nonce($_GET['state'], 'yourbeans_stripe_connect')) {
            add_settings_error('yourbeans_messages', 'invalid_nonce', __('Invalid request', 'yourbeans-coffee'), 'error');
            wp_redirect(admin_url('admin.php?page=yourbeans-settings'));
            exit;
        }

        // Exchange authorization code for access token
        $response = $this->exchange_code_for_token(sanitize_text_field($_GET['code']));

        if (is_wp_error($response)) {
            add_settings_error('yourbeans_messages', 'token_error', $response->get_error_message(), 'error');
            wp_redirect(admin_url('admin.php?page=yourbeans-settings'));
            exit;
        }

        // Save connected account details
        update_option('yourbeans_stripe_account_id', $response['stripe_user_id']);
        update_option('yourbeans_stripe_connected', true);
        update_option('yourbeans_stripe_connected_at', current_time('mysql'));

        // Log success
        error_log('YourBeans: Stripe Connect account connected - ' . $response['stripe_user_id']);

        add_settings_error(
            'yourbeans_messages',
            'stripe_connected',
            __('Stripe account connected successfully!', 'yourbeans-coffee'),
            'success'
        );

        wp_redirect(admin_url('admin.php?page=yourbeans-settings&stripe_connected=1'));
        exit;
    }

    /**
     * Exchange authorization code for access token
     *
     * @param string $code Authorization code
     * @return array|WP_Error Token data or error
     */
    private function exchange_code_for_token($code) {
        // This requires MyShop's Stripe secret key
        // In production, this would be done via MyShop API
        $response = wp_remote_post('https://myshop.coffee/wp-json/yourbeans/v1/stripe/connect', array(
            'body' => array(
                'code' => $code,
                'store_url' => home_url(),
            ),
            'headers' => array(
                'Authorization' => 'Bearer ' . get_option('yourbeans_api_key'),
            ),
            'timeout' => 30,
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $code = wp_remote_retrieve_response_code($response);

        if ($code >= 400) {
            return new WP_Error(
                'stripe_connect_failed',
                $body['message'] ?? __('Failed to connect Stripe account', 'yourbeans-coffee')
            );
        }

        return $body;
    }

    /**
     * Check if Stripe is connected
     *
     * @return bool
     */
    public function is_connected() {
        return !empty(get_option('yourbeans_stripe_account_id'));
    }

    /**
     * Get connected account ID
     *
     * @return string
     */
    public function get_account_id() {
        return get_option('yourbeans_stripe_account_id', '');
    }

    /**
     * Calculate platform fee for an order
     *
     * Platform fee = Sum of COGS for all YourBeans products + shipping + platform percentage
     *
     * @param WC_Order $order Order object
     * @return int Platform fee in cents
     */
    public function calculate_platform_fee($order) {
        $platform_fee = 0;

        // Calculate COGS for YourBeans products
        foreach ($order->get_items() as $item) {
            $product_id = $item->get_product_id();
            $variation_id = $item->get_variation_id();

            // Check if this is a YourBeans product
            $myshop_product_id = get_post_meta($product_id, '_yourbeans_myshop_product_id', true);

            if (!$myshop_product_id) {
                continue;
            }

            // Get COGS for this variant
            $cogs = floatval(get_post_meta($variation_id ?: $product_id, '_yourbeans_cogs', true));
            $quantity = $item->get_quantity();

            $platform_fee += ($cogs * $quantity);
        }

        // If no YourBeans products, no fee
        if ($platform_fee === 0) {
            return 0;
        }

        // Add shipping cost (MyShop handles fulfillment, so they need shipping costs)
        $platform_fee += floatval($order->get_shipping_total());

        // Add platform fee percentage (e.g., 2% of total)
        $platform_percentage = floatval(get_option('yourbeans_platform_fee_percent', 2)) / 100;
        $platform_fee += ($order->get_total() * $platform_percentage);

        // Convert to cents
        return (int) round($platform_fee * 100);
    }

    /**
     * Check if order contains YourBeans products
     *
     * @param WC_Order $order Order object
     * @return bool
     */
    public function order_has_yourbeans_products($order) {
        foreach ($order->get_items() as $item) {
            $product_id = $item->get_product_id();
            if (get_post_meta($product_id, '_yourbeans_myshop_product_id', true)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Add application fee to Stripe payment intent
     *
     * This hooks into WooCommerce Stripe Gateway
     *
     * @param array    $args Payment intent arguments
     * @param WC_Order $order Order object
     * @return array Modified arguments
     */
    public function add_application_fee($args, $order) {
        // Check if Stripe Connect is configured
        $stripe_account_id = $this->get_account_id();

        if (empty($stripe_account_id)) {
            return $args;
        }

        // Check if order has YourBeans products
        if (!$this->order_has_yourbeans_products($order)) {
            return $args;
        }

        // Calculate platform fee
        $platform_fee = $this->calculate_platform_fee($order);

        if ($platform_fee <= 0) {
            return $args;
        }

        // For Direct Charges model:
        // - Payment goes through connected account
        // - Platform takes application fee
        $args['application_fee_amount'] = $platform_fee;

        // Store the platform fee on the order for reference
        $order->update_meta_data('_yourbeans_platform_fee', $platform_fee / 100);
        $order->update_meta_data('_yourbeans_stripe_split_enabled', true);
        $order->save();

        error_log(sprintf(
            'YourBeans: Added application fee of $%.2f to order %d',
            $platform_fee / 100,
            $order->get_id()
        ));

        return $args;
    }

    /**
     * Modify Stripe payment request for destination charges
     *
     * Alternative approach using destination charges
     *
     * @param array      $request Payment request
     * @param WC_Order   $order Order object
     * @param object     $source Source object
     * @return array Modified request
     */
    public function modify_stripe_request($request, $order, $source) {
        // This is an alternative approach if using destination charges
        // Uncomment if preferred over application_fee_amount approach

        /*
        $stripe_account_id = $this->get_account_id();

        if (empty($stripe_account_id) || !$this->order_has_yourbeans_products($order)) {
            return $request;
        }

        $platform_fee = $this->calculate_platform_fee($order);

        if ($platform_fee > 0) {
            $request['transfer_data'] = array(
                'destination' => $stripe_account_id,
            );
            $request['application_fee_amount'] = $platform_fee;
        }
        */

        return $request;
    }

    /**
     * Disconnect Stripe account
     */
    public function disconnect() {
        delete_option('yourbeans_stripe_account_id');
        delete_option('yourbeans_stripe_connected');
        delete_option('yourbeans_stripe_connected_at');

        error_log('YourBeans: Stripe Connect account disconnected');
    }

    /**
     * AJAX: Disconnect Stripe
     */
    public function ajax_disconnect_stripe() {
        check_ajax_referer('yourbeans_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Permission denied', 'yourbeans-coffee')));
        }

        $this->disconnect();

        wp_send_json_success(array('message' => __('Stripe account disconnected', 'yourbeans-coffee')));
    }

    /**
     * Get connection status for display
     *
     * @return array Status information
     */
    public function get_status() {
        return array(
            'connected' => $this->is_connected(),
            'account_id' => $this->get_account_id(),
            'connected_at' => get_option('yourbeans_stripe_connected_at', ''),
        );
    }
}
