<?php
/**
 * YourBeans Shipping Class
 *
 * Calculates shipping costs for YourBeans products based on zone and weight.
 * Shipping rates are fetched from MyShop.Coffee API and cached.
 *
 * @package YourBeans_Coffee
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class YourBeans_Shipping
 */
class YourBeans_Shipping {

    /**
     * Shipping zones and rates
     *
     * @var array
     */
    private $shipping_zones = array();

    /**
     * Cache key for shipping rates
     *
     * @var string
     */
    private $cache_key = 'yourbeans_shipping_rates';

    /**
     * Cache duration in seconds (24 hours)
     *
     * @var int
     */
    private $cache_duration = DAY_IN_SECONDS;

    /**
     * Metro postcodes by state
     *
     * @var array
     */
    private $metro_postcodes = array(
        'NSW' => array(
            array('start' => 2000, 'end' => 2249),
            array('start' => 2760, 'end' => 2770),
        ),
        'VIC' => array(
            array('start' => 3000, 'end' => 3207),
            array('start' => 3335, 'end' => 3341),
            array('start' => 3427, 'end' => 3429),
            array('start' => 3750, 'end' => 3810),
        ),
        'QLD' => array(
            array('start' => 4000, 'end' => 4179),
            array('start' => 4205, 'end' => 4230),
        ),
        'SA' => array(
            array('start' => 5000, 'end' => 5199),
        ),
        'ACT' => array(
            array('start' => 2600, 'end' => 2618),
            array('start' => 2900, 'end' => 2920),
        ),
        'WA' => array(
            array('start' => 6000, 'end' => 6199),
        ),
    );

    /**
     * Constructor
     */
    public function __construct() {
        $this->load_shipping_rates();
    }

    /**
     * Load shipping rates from MyShop API or cache
     */
    private function load_shipping_rates() {
        $cached = get_transient($this->cache_key);

        if ($cached !== false) {
            $this->shipping_zones = $cached;
            return;
        }

        try {
            // Try to fetch from API
            if (function_exists('yourbeans') && yourbeans()->api) {
                $rates = yourbeans()->api->get_shipping_rates();
                if (!empty($rates)) {
                    $this->shipping_zones = $rates;
                    set_transient($this->cache_key, $rates, $this->cache_duration);
                    return;
                }
            }
        } catch (Exception $e) {
            error_log('YourBeans Shipping: Failed to fetch rates from API - ' . $e->getMessage());
        }

        // Fallback to default rates
        $this->shipping_zones = $this->get_default_rates();
        set_transient($this->cache_key, $this->shipping_zones, $this->cache_duration);
    }

    /**
     * Calculate shipping cost for an order
     *
     * @param WC_Order $order Order object
     * @return float Shipping cost
     */
    public function calculate_shipping_cost($order) {
        $shipping_state = $order->get_shipping_state();
        $shipping_postcode = $order->get_shipping_postcode();

        // Fall back to billing address if no shipping address
        if (empty($shipping_state)) {
            $shipping_state = $order->get_billing_state();
        }
        if (empty($shipping_postcode)) {
            $shipping_postcode = $order->get_billing_postcode();
        }

        $zone = $this->determine_zone($shipping_state, $shipping_postcode);
        $total_weight = $this->calculate_order_weight($order);

        return $this->get_zone_rate($zone, $total_weight);
    }

    /**
     * Calculate shipping cost for cart (YourBeans items only)
     *
     * @param WC_Cart $cart Cart object
     * @param string  $state State code
     * @param string  $postcode Postcode
     * @return float Shipping cost
     */
    public function calculate_cart_shipping($cart, $state, $postcode) {
        $zone = $this->determine_zone($state, $postcode);
        $total_weight = $this->calculate_cart_weight($cart);

        if ($total_weight <= 0) {
            return 0;
        }

        return $this->get_zone_rate($zone, $total_weight);
    }

    /**
     * Calculate total weight of YourBeans items in order
     *
     * @param WC_Order $order Order object
     * @return float Weight in kg
     */
    private function calculate_order_weight($order) {
        $total_weight_grams = 0;

        foreach ($order->get_items() as $item) {
            $product_id = $item->get_product_id();

            // Skip non-YourBeans items
            if (!get_post_meta($product_id, '_yourbeans_myshop_product_id', true)) {
                continue;
            }

            $variation_id = $item->get_variation_id();
            $size = '';

            if ($variation_id) {
                $size = get_post_meta($variation_id, '_yourbeans_size', true);
            }

            if (empty($size)) {
                // Try to determine from product attributes
                $product = $item->get_product();
                if ($product && $product->is_type('variation')) {
                    $attributes = $product->get_attributes();
                    $size = isset($attributes['size']) ? $attributes['size'] : '250g';
                } else {
                    $size = '250g'; // Default
                }
            }

            $quantity = $item->get_quantity();
            $weight = ($size === '1kg') ? 1000 : 250;
            $total_weight_grams += ($weight * $quantity);
        }

        return $total_weight_grams / 1000; // Return in kg
    }

    /**
     * Calculate total weight of YourBeans items in cart
     *
     * @param WC_Cart $cart Cart object
     * @return float Weight in kg
     */
    private function calculate_cart_weight($cart) {
        $total_weight_grams = 0;

        foreach ($cart->get_cart() as $cart_item) {
            $product_id = $cart_item['product_id'];

            // Skip non-YourBeans items
            if (!get_post_meta($product_id, '_yourbeans_myshop_product_id', true)) {
                continue;
            }

            $variation_id = isset($cart_item['variation_id']) ? $cart_item['variation_id'] : 0;
            $size = '';

            if ($variation_id) {
                $size = get_post_meta($variation_id, '_yourbeans_size', true);
            }

            if (empty($size)) {
                $product = $cart_item['data'];
                if ($product && $product->is_type('variation')) {
                    $attributes = $product->get_attributes();
                    $size = isset($attributes['size']) ? $attributes['size'] : '250g';
                } else {
                    $size = '250g';
                }
            }

            $quantity = $cart_item['quantity'];
            $weight = ($size === '1kg') ? 1000 : 250;
            $total_weight_grams += ($weight * $quantity);
        }

        return $total_weight_grams / 1000;
    }

    /**
     * Determine shipping zone from address
     *
     * @param string $state State code (e.g., 'NSW', 'VIC')
     * @param string $postcode Postcode
     * @return string Zone name (metro, regional, remote)
     */
    public function determine_zone($state, $postcode) {
        $state = strtoupper($state);
        $postcode = intval($postcode);

        // Remote states
        $remote_states = array('NT');

        if (in_array($state, $remote_states)) {
            return 'remote';
        }

        // Tasmania is regional or remote depending on postcode
        if ($state === 'TAS') {
            // Hobart metro area
            if ($postcode >= 7000 && $postcode <= 7099) {
                return 'regional';
            }
            return 'remote';
        }

        // WA - most is remote except Perth metro
        if ($state === 'WA') {
            if ($this->is_metro_postcode($postcode, $state)) {
                return 'regional'; // Perth metro treated as regional due to distance
            }
            return 'remote';
        }

        // Check if metro postcode
        if ($this->is_metro_postcode($postcode, $state)) {
            return 'metro';
        }

        return 'regional';
    }

    /**
     * Check if postcode is in metro area for a state
     *
     * @param int    $postcode Postcode
     * @param string $state State code
     * @return bool
     */
    private function is_metro_postcode($postcode, $state) {
        if (!isset($this->metro_postcodes[$state])) {
            return false;
        }

        foreach ($this->metro_postcodes[$state] as $range) {
            if ($postcode >= $range['start'] && $postcode <= $range['end']) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get shipping rate for zone and weight
     *
     * @param string $zone Zone name
     * @param float  $weight_kg Weight in kg
     * @return float Shipping cost
     */
    private function get_zone_rate($zone, $weight_kg) {
        $rates = isset($this->shipping_zones[$zone])
            ? $this->shipping_zones[$zone]
            : $this->shipping_zones['regional'];

        $base_rate = floatval($rates['base']);
        $per_kg_rate = floatval($rates['per_kg']);

        // First kg is included in base rate
        $extra_kg = max(0, ceil($weight_kg) - 1);

        return $base_rate + ($extra_kg * $per_kg_rate);
    }

    /**
     * Get default shipping rates (fallback)
     *
     * @return array Default shipping rates
     */
    private function get_default_rates() {
        return array(
            'metro' => array(
                'base' => 8.50,
                'per_kg' => 2.00,
                'label' => __('Metro (Sydney, Melbourne, Brisbane, Adelaide)', 'yourbeans-coffee'),
            ),
            'regional' => array(
                'base' => 10.50,
                'per_kg' => 2.50,
                'label' => __('Regional', 'yourbeans-coffee'),
            ),
            'remote' => array(
                'base' => 14.50,
                'per_kg' => 3.00,
                'label' => __('Remote (WA, NT, TAS)', 'yourbeans-coffee'),
            ),
        );
    }

    /**
     * Get all shipping zones with rates
     *
     * @return array Shipping zones
     */
    public function get_shipping_zones() {
        return $this->shipping_zones;
    }

    /**
     * Refresh shipping rates from API
     *
     * @return bool Success
     */
    public function refresh_rates() {
        delete_transient($this->cache_key);
        $this->load_shipping_rates();
        return true;
    }

    /**
     * Get estimated shipping cost for display
     *
     * @param string $zone Zone name
     * @param string $size Product size (250g or 1kg)
     * @param int    $quantity Quantity
     * @return array Shipping estimate
     */
    public function get_shipping_estimate($zone, $size = '250g', $quantity = 1) {
        $weight = ($size === '1kg') ? $quantity : ($quantity * 0.25);
        $cost = $this->get_zone_rate($zone, $weight);

        return array(
            'zone' => $zone,
            'zone_label' => isset($this->shipping_zones[$zone]['label'])
                ? $this->shipping_zones[$zone]['label']
                : ucfirst($zone),
            'weight_kg' => $weight,
            'cost' => $cost,
            'formatted_cost' => wc_price($cost),
        );
    }

    /**
     * Display shipping rates table (for admin)
     *
     * @return string HTML table
     */
    public function get_rates_table_html() {
        $html = '<table class="widefat yourbeans-shipping-rates">';
        $html .= '<thead><tr>';
        $html .= '<th>' . esc_html__('Zone', 'yourbeans-coffee') . '</th>';
        $html .= '<th>' . esc_html__('Base Rate', 'yourbeans-coffee') . '</th>';
        $html .= '<th>' . esc_html__('Per Additional kg', 'yourbeans-coffee') . '</th>';
        $html .= '<th>' . esc_html__('Example: 1kg', 'yourbeans-coffee') . '</th>';
        $html .= '<th>' . esc_html__('Example: 2kg', 'yourbeans-coffee') . '</th>';
        $html .= '</tr></thead>';
        $html .= '<tbody>';

        foreach ($this->shipping_zones as $zone => $rates) {
            $label = isset($rates['label']) ? $rates['label'] : ucfirst($zone);
            $base = floatval($rates['base']);
            $per_kg = floatval($rates['per_kg']);

            $html .= '<tr>';
            $html .= '<td><strong>' . esc_html($label) . '</strong></td>';
            $html .= '<td>' . wc_price($base) . '</td>';
            $html .= '<td>' . wc_price($per_kg) . '</td>';
            $html .= '<td>' . wc_price($base) . '</td>';
            $html .= '<td>' . wc_price($base + $per_kg) . '</td>';
            $html .= '</tr>';
        }

        $html .= '</tbody></table>';
        $html .= '<p class="description">' . esc_html__('Shipping rates are set by MyShop.Coffee and cannot be modified.', 'yourbeans-coffee') . '</p>';

        return $html;
    }
}
