<?php
/**
 * YourBeans Settings
 *
 * Handles plugin settings and options
 *
 * @package YourBeans_Coffee
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class YourBeans_Settings
 */
class YourBeans_Settings {

    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_init', array($this, 'register_settings'));
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        // API Settings Section
        add_settings_section(
            'yourbeans_api_settings',
            __('API Configuration', 'yourbeans-coffee'),
            array($this, 'api_section_callback'),
            'yourbeans_settings'
        );

        // API Key
        register_setting('yourbeans_settings', 'yourbeans_api_key', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => '',
        ));

        add_settings_field(
            'yourbeans_api_key',
            __('MyShop.Coffee API Key', 'yourbeans-coffee'),
            array($this, 'api_key_field_callback'),
            'yourbeans_settings',
            'yourbeans_api_settings'
        );

        // Product Settings Section
        add_settings_section(
            'yourbeans_product_settings',
            __('Product Settings', 'yourbeans-coffee'),
            array($this, 'product_section_callback'),
            'yourbeans_settings'
        );

        // Default Markup
        register_setting('yourbeans_settings', 'yourbeans_default_markup', array(
            'type' => 'integer',
            'sanitize_callback' => 'absint',
            'default' => 50,
        ));

        add_settings_field(
            'yourbeans_default_markup',
            __('Default Markup %', 'yourbeans-coffee'),
            array($this, 'markup_field_callback'),
            'yourbeans_settings',
            'yourbeans_product_settings'
        );

        // Auto-sync inventory
        register_setting('yourbeans_settings', 'yourbeans_auto_sync', array(
            'type' => 'boolean',
            'sanitize_callback' => 'rest_sanitize_boolean',
            'default' => true,
        ));

        add_settings_field(
            'yourbeans_auto_sync',
            __('Auto-sync Inventory', 'yourbeans-coffee'),
            array($this, 'auto_sync_field_callback'),
            'yourbeans_settings',
            'yourbeans_product_settings'
        );

        // Payment Settings Section
        add_settings_section(
            'yourbeans_payment_settings',
            __('Payment Settings', 'yourbeans-coffee'),
            array($this, 'payment_section_callback'),
            'yourbeans_settings'
        );

        // Platform Fee Percentage
        register_setting('yourbeans_settings', 'yourbeans_platform_fee_percent', array(
            'type' => 'number',
            'sanitize_callback' => array($this, 'sanitize_percentage'),
            'default' => 2,
        ));

        add_settings_field(
            'yourbeans_platform_fee_percent',
            __('Platform Fee %', 'yourbeans-coffee'),
            array($this, 'platform_fee_field_callback'),
            'yourbeans_settings',
            'yourbeans_payment_settings'
        );
    }

    /**
     * API section description
     */
    public function api_section_callback() {
        echo '<p>' . esc_html__('Configure your connection to MyShop.Coffee', 'yourbeans-coffee') . '</p>';
    }

    /**
     * Product section description
     */
    public function product_section_callback() {
        echo '<p>' . esc_html__('Configure default settings for imported products', 'yourbeans-coffee') . '</p>';
    }

    /**
     * Payment section description
     */
    public function payment_section_callback() {
        echo '<p>' . esc_html__('Configure payment splitting via Stripe Connect', 'yourbeans-coffee') . '</p>';
    }

    /**
     * API key field
     */
    public function api_key_field_callback() {
        $value = get_option('yourbeans_api_key', '');
        ?>
        <input type="text"
               name="yourbeans_api_key"
               value="<?php echo esc_attr($value); ?>"
               class="regular-text"
               autocomplete="off" />
        <p class="description">
            <?php esc_html_e('Get this from your MyShop.Coffee vendor dashboard under API Settings.', 'yourbeans-coffee'); ?>
        </p>
        <?php
    }

    /**
     * Markup field
     */
    public function markup_field_callback() {
        $value = get_option('yourbeans_default_markup', 50);
        ?>
        <input type="number"
               name="yourbeans_default_markup"
               value="<?php echo esc_attr($value); ?>"
               min="0"
               max="500"
               step="5"
               class="small-text" /> %
        <p class="description">
            <?php esc_html_e('Default markup percentage applied to imported products. Can be overridden per product.', 'yourbeans-coffee'); ?>
        </p>
        <?php
    }

    /**
     * Auto-sync field
     */
    public function auto_sync_field_callback() {
        $value = get_option('yourbeans_auto_sync', true);
        ?>
        <label>
            <input type="checkbox"
                   name="yourbeans_auto_sync"
                   value="1"
                   <?php checked($value, true); ?> />
            <?php esc_html_e('Automatically sync inventory levels hourly', 'yourbeans-coffee'); ?>
        </label>
        <?php
    }

    /**
     * Platform fee field
     */
    public function platform_fee_field_callback() {
        $value = get_option('yourbeans_platform_fee_percent', 2);
        ?>
        <input type="number"
               name="yourbeans_platform_fee_percent"
               value="<?php echo esc_attr($value); ?>"
               min="0"
               max="10"
               step="0.5"
               class="small-text" /> %
        <p class="description">
            <?php esc_html_e('Additional platform fee percentage added to COGS + shipping for Stripe Connect splits.', 'yourbeans-coffee'); ?>
        </p>
        <?php
    }

    /**
     * Sanitize percentage value
     *
     * @param mixed $value Input value
     * @return float Sanitized value
     */
    public function sanitize_percentage($value) {
        $value = floatval($value);
        return max(0, min(100, $value));
    }

    /**
     * Get all settings
     *
     * @return array Settings array
     */
    public static function get_all() {
        return array(
            'api_key' => get_option('yourbeans_api_key', ''),
            'default_markup' => get_option('yourbeans_default_markup', 50),
            'auto_sync' => get_option('yourbeans_auto_sync', true),
            'platform_fee_percent' => get_option('yourbeans_platform_fee_percent', 2),
            'stripe_account_id' => get_option('yourbeans_stripe_account_id', ''),
            'stripe_connected' => get_option('yourbeans_stripe_connected', false),
            'last_sync' => get_option('yourbeans_last_sync', ''),
        );
    }
}
