<?php
/**
 * YourBeans Products Handler
 *
 * Handles product import and synchronization
 *
 * @package YourBeans_Coffee
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class YourBeans_Products
 */
class YourBeans_Products {

    /**
     * API instance
     *
     * @var YourBeans_API
     */
    private $api;

    /**
     * Available sizes
     *
     * @var array
     */
    private $sizes = array('250g', '1kg');

    /**
     * Available grind types
     *
     * @var array
     */
    private $grinds = array(
        'Whole Bean',
        'Espresso',
        'Filter',
        'Plunger',
        'Stovetop',
        'Aeropress',
    );

    /**
     * Constructor
     *
     * @param YourBeans_API $api API instance
     */
    public function __construct($api) {
        $this->api = $api;

        // AJAX handlers
        add_action('wp_ajax_yourbeans_import_product', array($this, 'ajax_import_product'));
        add_action('wp_ajax_yourbeans_get_catalog', array($this, 'ajax_get_catalog'));
    }

    /**
     * Import a product from MyShop.Coffee
     *
     * @param int   $myshop_product_id MyShop product ID
     * @param array $options Import options (title, markup, etc.)
     * @return int|WP_Error WooCommerce product ID or error
     */
    public function import_product($myshop_product_id, $options = array()) {
        try {
            // Fetch product from MyShop API
            $myshop_product = $this->api->get_product($myshop_product_id);

            if (empty($myshop_product)) {
                return new WP_Error('product_not_found', __('Product not found in MyShop catalog', 'yourbeans-coffee'));
            }

            // Check if already imported
            $existing = $this->get_product_by_myshop_id($myshop_product_id);
            if ($existing) {
                return new WP_Error('already_imported', __('Product already imported', 'yourbeans-coffee'), array('product_id' => $existing));
            }

            // Create WooCommerce variable product
            $product = new WC_Product_Variable();

            // Set basic details
            $product->set_name($options['title'] ?? $myshop_product['title']);
            $product->set_description($myshop_product['description'] ?? '');
            $product->set_short_description($myshop_product['short_description'] ?? '');
            $product->set_status('publish');
            $product->set_catalog_visibility('visible');
            $product->set_manage_stock(false); // Managed per-variant

            // Set product attributes (Size and Grind)
            $this->set_product_attributes($product);

            // Save to get ID
            $product_id = $product->save();

            if (!$product_id) {
                return new WP_Error('save_failed', __('Failed to create product', 'yourbeans-coffee'));
            }

            // Store MyShop reference
            update_post_meta($product_id, '_yourbeans_myshop_product_id', $myshop_product_id);
            update_post_meta($product_id, '_yourbeans_cogs_base', $myshop_product['base_price'] ?? 0);
            update_post_meta($product_id, '_yourbeans_imported_at', current_time('mysql'));

            // Create all 12 variants
            $this->create_variants($product_id, $myshop_product, $options);

            // Import images
            if (!empty($myshop_product['images'])) {
                $this->import_images($product_id, $myshop_product['images']);
            }

            // Log success
            error_log(sprintf(
                'YourBeans: Imported product %d from MyShop product %d',
                $product_id,
                $myshop_product_id
            ));

            return $product_id;

        } catch (Exception $e) {
            error_log('YourBeans import error: ' . $e->getMessage());
            return new WP_Error('import_failed', $e->getMessage());
        }
    }

    /**
     * Set product attributes (Size and Grind)
     *
     * @param WC_Product_Variable $product Product object
     */
    private function set_product_attributes($product) {
        $attributes = array();

        // Size attribute
        $size_attr = new WC_Product_Attribute();
        $size_attr->set_id(0);
        $size_attr->set_name('Size');
        $size_attr->set_options($this->sizes);
        $size_attr->set_position(0);
        $size_attr->set_visible(true);
        $size_attr->set_variation(true);
        $attributes[] = $size_attr;

        // Grind attribute
        $grind_attr = new WC_Product_Attribute();
        $grind_attr->set_id(0);
        $grind_attr->set_name('Grind');
        $grind_attr->set_options($this->grinds);
        $grind_attr->set_position(1);
        $grind_attr->set_visible(true);
        $grind_attr->set_variation(true);
        $attributes[] = $grind_attr;

        $product->set_attributes($attributes);
    }

    /**
     * Create all 12 variants (2 sizes × 6 grinds)
     *
     * @param int   $product_id WooCommerce product ID
     * @param array $myshop_product MyShop product data
     * @param array $options Import options
     */
    private function create_variants($product_id, $myshop_product, $options) {
        $markup = floatval($options['markup'] ?? get_option('yourbeans_default_markup', 50)) / 100;
        $base_price = floatval($myshop_product['base_price'] ?? 0);

        foreach ($this->sizes as $size) {
            foreach ($this->grinds as $grind) {
                $variation = new WC_Product_Variation();
                $variation->set_parent_id($product_id);

                // Calculate price
                // 1kg = 3.8× the 250g price
                $size_multiplier = ($size === '1kg') ? 3.8 : 1;
                $cogs = $base_price * $size_multiplier;
                $retail_price = $cogs * (1 + $markup);

                $variation->set_regular_price(number_format($retail_price, 2, '.', ''));

                // Set attributes
                $variation->set_attributes(array(
                    'size' => $size,
                    'grind' => sanitize_title($grind),
                ));

                // Generate SKU: YB-{myshop_id}-{size}-{grind_code}
                $grind_code = strtoupper(substr(str_replace(' ', '', $grind), 0, 4));
                $size_code = str_replace('g', '', $size);
                $sku = sprintf('YB-%d-%s-%s', $myshop_product['id'], $size_code, $grind_code);
                $variation->set_sku($sku);

                // Stock management
                $variation->set_manage_stock(true);
                $variation->set_stock_quantity($myshop_product['stock'] ?? 100);
                $variation->set_stock_status('instock');

                // Save variation
                $variation_id = $variation->save();

                // Store COGS for Stripe Connect fee calculation
                update_post_meta($variation_id, '_yourbeans_cogs', $cogs);
                update_post_meta($variation_id, '_yourbeans_myshop_product_id', $myshop_product['id']);
            }
        }

        // Clear transients
        wc_delete_product_transients($product_id);
    }

    /**
     * Import product images
     *
     * @param int   $product_id Product ID
     * @param array $images Array of image URLs
     */
    private function import_images($product_id, $images) {
        if (empty($images)) {
            return;
        }

        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';

        $gallery_ids = array();

        foreach ($images as $index => $image_url) {
            // Download and attach image
            $attachment_id = media_sideload_image($image_url, $product_id, '', 'id');

            if (!is_wp_error($attachment_id)) {
                if ($index === 0) {
                    // Set first image as featured
                    set_post_thumbnail($product_id, $attachment_id);
                } else {
                    $gallery_ids[] = $attachment_id;
                }
            }
        }

        // Set gallery images
        if (!empty($gallery_ids)) {
            update_post_meta($product_id, '_product_image_gallery', implode(',', $gallery_ids));
        }
    }

    /**
     * Update inventory for a product
     *
     * @param int $myshop_product_id MyShop product ID
     * @param int $stock New stock quantity
     */
    public function update_inventory($myshop_product_id, $stock) {
        global $wpdb;

        // Find all variations with this MyShop product ID
        $variation_ids = $wpdb->get_col($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta}
             WHERE meta_key = '_yourbeans_myshop_product_id'
             AND meta_value = %d",
            $myshop_product_id
        ));

        foreach ($variation_ids as $variation_id) {
            $variation = wc_get_product($variation_id);
            if ($variation && $variation->is_type('variation')) {
                $variation->set_stock_quantity($stock);
                $variation->set_stock_status($stock > 0 ? 'instock' : 'outofstock');
                $variation->save();
            }
        }
    }

    /**
     * Get WooCommerce product ID by MyShop ID
     *
     * @param int $myshop_product_id MyShop product ID
     * @return int|null Product ID or null
     */
    public function get_product_by_myshop_id($myshop_product_id) {
        global $wpdb;

        return $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta}
             WHERE meta_key = '_yourbeans_myshop_product_id'
             AND meta_value = %d
             AND post_id IN (SELECT ID FROM {$wpdb->posts} WHERE post_type = 'product')",
            $myshop_product_id
        ));
    }

    /**
     * Get count of imported products
     *
     * @return int Count
     */
    public function get_imported_count() {
        global $wpdb;

        return (int) $wpdb->get_var(
            "SELECT COUNT(DISTINCT post_id) FROM {$wpdb->postmeta}
             WHERE meta_key = '_yourbeans_myshop_product_id'
             AND post_id IN (SELECT ID FROM {$wpdb->posts} WHERE post_type = 'product' AND post_status = 'publish')"
        );
    }

    /**
     * AJAX: Import product
     */
    public function ajax_import_product() {
        check_ajax_referer('yourbeans_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Permission denied', 'yourbeans-coffee')));
        }

        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
        $title = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : '';
        $markup = isset($_POST['markup']) ? floatval($_POST['markup']) : null;

        if (!$product_id) {
            wp_send_json_error(array('message' => __('Invalid product ID', 'yourbeans-coffee')));
        }

        $options = array();
        if ($title) {
            $options['title'] = $title;
        }
        if ($markup !== null) {
            $options['markup'] = $markup;
        }

        $result = $this->import_product($product_id, $options);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        wp_send_json_success(array(
            'message' => __('Product imported successfully!', 'yourbeans-coffee'),
            'product_id' => $result,
            'edit_url' => get_edit_post_link($result, 'raw'),
        ));
    }

    /**
     * AJAX: Get catalog
     */
    public function ajax_get_catalog() {
        check_ajax_referer('yourbeans_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Permission denied', 'yourbeans-coffee')));
        }

        try {
            $args = array(
                'page' => isset($_GET['page']) ? intval($_GET['page']) : 1,
                'per_page' => isset($_GET['per_page']) ? intval($_GET['per_page']) : 20,
                'search' => isset($_GET['search']) ? sanitize_text_field($_GET['search']) : '',
                'category' => isset($_GET['category']) ? sanitize_text_field($_GET['category']) : '',
            );

            $products = $this->api->get_products($args);

            // Mark which products are already imported
            foreach ($products as &$product) {
                $product['imported'] = $this->get_product_by_myshop_id($product['id']) ? true : false;
            }

            wp_send_json_success(array('products' => $products));

        } catch (Exception $e) {
            wp_send_json_error(array('message' => $e->getMessage()));
        }
    }
}
