<?php
/**
 * YourBeans Orders Handler
 *
 * Handles order submission and fulfillment
 *
 * @package YourBeans_Coffee
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class YourBeans_Orders
 */
class YourBeans_Orders {

    /**
     * API instance
     *
     * @var YourBeans_API
     */
    private $api;

    /**
     * Constructor
     *
     * @param YourBeans_API $api API instance
     */
    public function __construct($api) {
        $this->api = $api;

        // Hook into order status changes
        add_action('woocommerce_order_status_processing', array($this, 'submit_to_myshop'), 10, 1);
        add_action('woocommerce_order_status_completed', array($this, 'submit_to_myshop'), 10, 1);

        // Add tracking info to order emails
        add_action('woocommerce_email_order_meta', array($this, 'add_tracking_to_email'), 10, 3);

        // Add tracking info to order details
        add_action('woocommerce_order_details_after_order_table', array($this, 'display_tracking_info'), 10, 1);

        // Admin order meta box
        add_action('add_meta_boxes', array($this, 'add_order_meta_box'));
    }

    /**
     * Submit order to MyShop.Coffee for fulfillment
     *
     * @param int $order_id WooCommerce order ID
     */
    public function submit_to_myshop($order_id) {
        $order = wc_get_order($order_id);

        if (!$order) {
            return;
        }

        // Check if already submitted
        if ($order->get_meta('_yourbeans_submitted')) {
            return;
        }

        // Check if order contains YourBeans products
        $yourbeans_items = $this->get_yourbeans_items($order);

        if (empty($yourbeans_items)) {
            return; // No YourBeans products in this order
        }

        // Build order data for MyShop API
        $order_data = array(
            'external_order_id' => $order_id,
            'external_order_number' => $order->get_order_number(),
            'store_domain' => home_url(),
            'store_name' => get_bloginfo('name'),
            'customer' => array(
                'name' => $order->get_formatted_billing_full_name(),
                'email' => $order->get_billing_email(),
                'phone' => $order->get_billing_phone(),
            ),
            'shipping_address' => array(
                'name' => $order->get_formatted_shipping_full_name() ?: $order->get_formatted_billing_full_name(),
                'company' => $order->get_shipping_company(),
                'address_1' => $order->get_shipping_address_1() ?: $order->get_billing_address_1(),
                'address_2' => $order->get_shipping_address_2() ?: $order->get_billing_address_2(),
                'city' => $order->get_shipping_city() ?: $order->get_billing_city(),
                'state' => $order->get_shipping_state() ?: $order->get_billing_state(),
                'postcode' => $order->get_shipping_postcode() ?: $order->get_billing_postcode(),
                'country' => $order->get_shipping_country() ?: $order->get_billing_country(),
            ),
            'items' => $yourbeans_items,
            'shipping_method' => $order->get_shipping_method(),
            'order_notes' => $order->get_customer_note(),
            'created_at' => $order->get_date_created()->format('c'),
        );

        try {
            $response = $this->api->submit_order($order_data);

            // Mark as submitted
            $order->update_meta_data('_yourbeans_submitted', true);
            $order->update_meta_data('_yourbeans_submitted_at', current_time('mysql'));
            $order->update_meta_data('_yourbeans_order_id', $response['order_id'] ?? '');

            $order->add_order_note(sprintf(
                __('Order submitted to YourBeans Coffee for fulfillment. MyShop Order ID: %s', 'yourbeans-coffee'),
                $response['order_id'] ?? 'N/A'
            ));

            $order->save();

            error_log(sprintf(
                'YourBeans: Order %d submitted successfully. MyShop ID: %s',
                $order_id,
                $response['order_id'] ?? 'N/A'
            ));

        } catch (Exception $e) {
            $order->add_order_note(sprintf(
                __('Failed to submit to YourBeans Coffee: %s', 'yourbeans-coffee'),
                $e->getMessage()
            ));
            $order->save();

            error_log('YourBeans order submission failed for order ' . $order_id . ': ' . $e->getMessage());
        }
    }

    /**
     * Get YourBeans items from order
     *
     * @param WC_Order $order Order object
     * @return array YourBeans items
     */
    private function get_yourbeans_items($order) {
        $items = array();

        foreach ($order->get_items() as $item) {
            $product_id = $item->get_product_id();
            $variation_id = $item->get_variation_id();

            // Check if this is a YourBeans product
            $myshop_product_id = get_post_meta($product_id, '_yourbeans_myshop_product_id', true);

            if (!$myshop_product_id) {
                continue;
            }

            $variation = $variation_id ? wc_get_product($variation_id) : null;

            $items[] = array(
                'myshop_product_id' => $myshop_product_id,
                'sku' => $variation ? $variation->get_sku() : '',
                'name' => $item->get_name(),
                'quantity' => $item->get_quantity(),
                'size' => $variation ? $variation->get_attribute('size') : '',
                'grind' => $variation ? $variation->get_attribute('grind') : '',
                'subtotal' => $item->get_subtotal(),
                'cogs' => get_post_meta($variation_id ?: $product_id, '_yourbeans_cogs', true),
            );
        }

        return $items;
    }

    /**
     * Handle fulfillment webhook from MyShop
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error Response
     */
    public function handle_fulfillment_webhook($request) {
        $data = $request->get_json_params();

        $external_order_id = $data['external_order_id'] ?? null;
        $tracking_number = $data['tracking_number'] ?? null;
        $carrier = $data['carrier'] ?? '';
        $tracking_url = $data['tracking_url'] ?? '';
        $status = $data['status'] ?? '';
        $shipped_at = $data['shipped_at'] ?? '';

        if (!$external_order_id) {
            return new WP_Error('invalid_request', __('Missing order ID', 'yourbeans-coffee'), array('status' => 400));
        }

        $order = wc_get_order($external_order_id);

        if (!$order) {
            return new WP_Error('order_not_found', __('Order not found', 'yourbeans-coffee'), array('status' => 404));
        }

        // Update tracking information
        if ($tracking_number) {
            $order->update_meta_data('_yourbeans_tracking_number', $tracking_number);
            $order->update_meta_data('_yourbeans_carrier', $carrier);
            $order->update_meta_data('_yourbeans_tracking_url', $tracking_url);
            $order->update_meta_data('_yourbeans_shipped_at', $shipped_at);

            $order->add_order_note(sprintf(
                __('YourBeans shipment dispatched. Carrier: %s, Tracking: %s', 'yourbeans-coffee'),
                $carrier,
                $tracking_number
            ));
        }

        // Update order status based on fulfillment status
        switch ($status) {
            case 'shipped':
            case 'delivered':
                $order->update_status('completed', __('Order fulfilled by YourBeans Coffee', 'yourbeans-coffee'));
                break;

            case 'cancelled':
                $order->update_status('cancelled', __('Order cancelled by YourBeans Coffee', 'yourbeans-coffee'));
                break;
        }

        $order->save();

        error_log(sprintf(
            'YourBeans webhook: Order %d updated - Status: %s, Tracking: %s',
            $external_order_id,
            $status,
            $tracking_number
        ));

        return rest_ensure_response(array('success' => true));
    }

    /**
     * Add tracking info to order emails
     *
     * @param WC_Order $order Order object
     * @param bool     $sent_to_admin Sent to admin
     * @param bool     $plain_text Plain text email
     */
    public function add_tracking_to_email($order, $sent_to_admin, $plain_text) {
        $tracking_number = $order->get_meta('_yourbeans_tracking_number');
        $carrier = $order->get_meta('_yourbeans_carrier');
        $tracking_url = $order->get_meta('_yourbeans_tracking_url');

        if (!$tracking_number) {
            return;
        }

        if ($plain_text) {
            echo "\n" . __('Tracking Information', 'yourbeans-coffee') . "\n";
            echo __('Carrier:', 'yourbeans-coffee') . ' ' . $carrier . "\n";
            echo __('Tracking Number:', 'yourbeans-coffee') . ' ' . $tracking_number . "\n";
            if ($tracking_url) {
                echo __('Track your order:', 'yourbeans-coffee') . ' ' . $tracking_url . "\n";
            }
        } else {
            echo '<h2>' . __('Tracking Information', 'yourbeans-coffee') . '</h2>';
            echo '<p><strong>' . __('Carrier:', 'yourbeans-coffee') . '</strong> ' . esc_html($carrier) . '</p>';
            echo '<p><strong>' . __('Tracking Number:', 'yourbeans-coffee') . '</strong> ' . esc_html($tracking_number) . '</p>';
            if ($tracking_url) {
                echo '<p><a href="' . esc_url($tracking_url) . '" target="_blank">' . __('Track your order', 'yourbeans-coffee') . '</a></p>';
            }
        }
    }

    /**
     * Display tracking info on order details page
     *
     * @param WC_Order $order Order object
     */
    public function display_tracking_info($order) {
        $tracking_number = $order->get_meta('_yourbeans_tracking_number');
        $carrier = $order->get_meta('_yourbeans_carrier');
        $tracking_url = $order->get_meta('_yourbeans_tracking_url');

        if (!$tracking_number) {
            return;
        }

        ?>
        <h2><?php esc_html_e('Tracking Information', 'yourbeans-coffee'); ?></h2>
        <table class="woocommerce-table shop_table yourbeans-tracking">
            <tr>
                <th><?php esc_html_e('Carrier', 'yourbeans-coffee'); ?></th>
                <td><?php echo esc_html($carrier); ?></td>
            </tr>
            <tr>
                <th><?php esc_html_e('Tracking Number', 'yourbeans-coffee'); ?></th>
                <td>
                    <?php if ($tracking_url): ?>
                        <a href="<?php echo esc_url($tracking_url); ?>" target="_blank">
                            <?php echo esc_html($tracking_number); ?>
                        </a>
                    <?php else: ?>
                        <?php echo esc_html($tracking_number); ?>
                    <?php endif; ?>
                </td>
            </tr>
        </table>
        <?php
    }

    /**
     * Add order meta box in admin
     */
    public function add_order_meta_box() {
        $screen = class_exists('\Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController')
            && wc_get_container()->get(\Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController::class)->custom_orders_table_usage_is_enabled()
            ? wc_get_page_screen_id('shop-order')
            : 'shop_order';

        add_meta_box(
            'yourbeans-fulfillment',
            __('YourBeans Fulfillment', 'yourbeans-coffee'),
            array($this, 'render_order_meta_box'),
            $screen,
            'side',
            'default'
        );
    }

    /**
     * Render order meta box
     *
     * @param WP_Post|WC_Order $post_or_order Post or order object
     */
    public function render_order_meta_box($post_or_order) {
        $order = $post_or_order instanceof WC_Order ? $post_or_order : wc_get_order($post_or_order->ID);

        if (!$order) {
            return;
        }

        $submitted = $order->get_meta('_yourbeans_submitted');
        $myshop_order_id = $order->get_meta('_yourbeans_order_id');
        $tracking_number = $order->get_meta('_yourbeans_tracking_number');
        $carrier = $order->get_meta('_yourbeans_carrier');

        // Check if order has YourBeans products
        $has_yourbeans = !empty($this->get_yourbeans_items($order));

        if (!$has_yourbeans) {
            echo '<p>' . esc_html__('This order does not contain YourBeans products.', 'yourbeans-coffee') . '</p>';
            return;
        }

        ?>
        <div class="yourbeans-order-status">
            <p>
                <strong><?php esc_html_e('Status:', 'yourbeans-coffee'); ?></strong>
                <?php if ($submitted): ?>
                    <span class="status-submitted"><?php esc_html_e('Submitted', 'yourbeans-coffee'); ?></span>
                <?php else: ?>
                    <span class="status-pending"><?php esc_html_e('Pending', 'yourbeans-coffee'); ?></span>
                <?php endif; ?>
            </p>

            <?php if ($myshop_order_id): ?>
                <p>
                    <strong><?php esc_html_e('MyShop Order ID:', 'yourbeans-coffee'); ?></strong>
                    <?php echo esc_html($myshop_order_id); ?>
                </p>
            <?php endif; ?>

            <?php if ($tracking_number): ?>
                <p>
                    <strong><?php esc_html_e('Tracking:', 'yourbeans-coffee'); ?></strong>
                    <?php echo esc_html($carrier . ' - ' . $tracking_number); ?>
                </p>
            <?php endif; ?>

            <?php if (!$submitted): ?>
                <button type="button" class="button yourbeans-submit-order" data-order-id="<?php echo esc_attr($order->get_id()); ?>">
                    <?php esc_html_e('Submit to YourBeans', 'yourbeans-coffee'); ?>
                </button>
            <?php endif; ?>
        </div>
        <?php
    }
}
