<?php
/**
 * YourBeans Licensing Class
 *
 * Validates store owner's subscription to the plugin via MyShop's Stripe Billing.
 * Handles license verification, caching, and feature restrictions.
 *
 * @package YourBeans_Coffee
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class YourBeans_Licensing
 */
class YourBeans_Licensing {

    /**
     * API endpoint for license verification
     *
     * @var string
     */
    private $api_endpoint = 'https://myshop.coffee/wp-json/yourbeans/v1/license';

    /**
     * Cache transient key
     *
     * @var string
     */
    private $cache_key = 'yourbeans_license_status';

    /**
     * Cache duration in seconds (1 hour)
     *
     * @var int
     */
    private $cache_duration = HOUR_IN_SECONDS;

    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_init', array($this, 'maybe_restrict_features'));
        add_action('wp_ajax_yourbeans_activate_license', array($this, 'ajax_activate_license'));
        add_action('wp_ajax_yourbeans_deactivate_license', array($this, 'ajax_deactivate_license'));
        add_action('wp_ajax_yourbeans_refresh_license', array($this, 'ajax_refresh_license'));
    }

    /**
     * Check if store has active plugin subscription
     *
     * @param bool $force_refresh Force a fresh check from API
     * @return bool True if license is active
     */
    public function is_license_active($force_refresh = false) {
        if (!$force_refresh) {
            $cached = get_transient($this->cache_key);
            if ($cached !== false) {
                return $cached === 'active';
            }
        }

        $status = $this->verify_license();
        set_transient($this->cache_key, $status, $this->cache_duration);

        return $status === 'active';
    }

    /**
     * Get current license status details
     *
     * @return array License status information
     */
    public function get_license_status() {
        $license_key = get_option('yourbeans_license_key');

        if (empty($license_key)) {
            return array(
                'status' => 'not_configured',
                'message' => __('No license key configured.', 'yourbeans-coffee'),
                'is_active' => false,
            );
        }

        $cached_status = get_transient($this->cache_key);

        if ($cached_status === false) {
            $cached_status = $this->verify_license();
            set_transient($this->cache_key, $cached_status, $this->cache_duration);
        }

        $status_messages = array(
            'active' => __('Your subscription is active.', 'yourbeans-coffee'),
            'expired' => __('Your subscription has expired.', 'yourbeans-coffee'),
            'cancelled' => __('Your subscription has been cancelled.', 'yourbeans-coffee'),
            'past_due' => __('Your payment is past due. Please update your payment method.', 'yourbeans-coffee'),
            'invalid' => __('Invalid license key.', 'yourbeans-coffee'),
            'error' => __('Unable to verify license. Please try again later.', 'yourbeans-coffee'),
        );

        return array(
            'status' => $cached_status,
            'message' => isset($status_messages[$cached_status]) ? $status_messages[$cached_status] : $status_messages['error'],
            'is_active' => $cached_status === 'active',
            'license_key' => $this->mask_license_key($license_key),
        );
    }

    /**
     * Verify license with MyShop API
     *
     * @return string License status (active, expired, cancelled, past_due, invalid, error)
     */
    private function verify_license() {
        $license_key = get_option('yourbeans_license_key');

        if (empty($license_key)) {
            return 'invalid';
        }

        $store_url = home_url();
        $store_email = get_option('admin_email');

        $response = wp_remote_post($this->api_endpoint . '/verify', array(
            'body' => array(
                'license_key' => $license_key,
                'store_url' => $store_url,
                'store_email' => $store_email,
            ),
            'timeout' => 15,
            'sslverify' => true,
        ));

        if (is_wp_error($response)) {
            error_log('YourBeans License verification failed: ' . $response->get_error_message());
            return 'error';
        }

        $response_code = wp_remote_retrieve_response_code($response);

        if ($response_code !== 200) {
            error_log('YourBeans License verification failed with status: ' . $response_code);
            return 'error';
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!isset($data['status'])) {
            return 'error';
        }

        // Store subscription details for display
        if (isset($data['subscription'])) {
            update_option('yourbeans_subscription_details', $data['subscription']);
        }

        return sanitize_text_field($data['status']);
    }

    /**
     * Activate a license key
     *
     * @param string $license_key The license key to activate
     * @return array Result with success status and message
     */
    public function activate_license($license_key) {
        $license_key = sanitize_text_field($license_key);

        if (empty($license_key)) {
            return array(
                'success' => false,
                'message' => __('Please enter a license key.', 'yourbeans-coffee'),
            );
        }

        $store_url = home_url();
        $store_email = get_option('admin_email');

        $response = wp_remote_post($this->api_endpoint . '/activate', array(
            'body' => array(
                'license_key' => $license_key,
                'store_url' => $store_url,
                'store_email' => $store_email,
                'store_name' => get_bloginfo('name'),
            ),
            'timeout' => 15,
            'sslverify' => true,
        ));

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => __('Unable to connect to license server. Please try again.', 'yourbeans-coffee'),
            );
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($data['success']) && $data['success']) {
            update_option('yourbeans_license_key', $license_key);
            delete_transient($this->cache_key);

            return array(
                'success' => true,
                'message' => __('License activated successfully!', 'yourbeans-coffee'),
            );
        }

        return array(
            'success' => false,
            'message' => isset($data['message']) ? $data['message'] : __('Invalid license key.', 'yourbeans-coffee'),
        );
    }

    /**
     * Deactivate the current license
     *
     * @return array Result with success status and message
     */
    public function deactivate_license() {
        $license_key = get_option('yourbeans_license_key');

        if (empty($license_key)) {
            return array(
                'success' => false,
                'message' => __('No license key to deactivate.', 'yourbeans-coffee'),
            );
        }

        $response = wp_remote_post($this->api_endpoint . '/deactivate', array(
            'body' => array(
                'license_key' => $license_key,
                'store_url' => home_url(),
            ),
            'timeout' => 15,
            'sslverify' => true,
        ));

        // Clear local data regardless of remote response
        delete_option('yourbeans_license_key');
        delete_option('yourbeans_subscription_details');
        delete_transient($this->cache_key);

        if (is_wp_error($response)) {
            return array(
                'success' => true,
                'message' => __('License deactivated locally. Remote deactivation may have failed.', 'yourbeans-coffee'),
            );
        }

        return array(
            'success' => true,
            'message' => __('License deactivated successfully.', 'yourbeans-coffee'),
        );
    }

    /**
     * Get subscription management URL (Stripe Customer Portal)
     *
     * @return string|false Billing portal URL or false on failure
     */
    public function get_billing_portal_url() {
        $license_key = get_option('yourbeans_license_key');

        if (empty($license_key)) {
            return false;
        }

        $response = wp_remote_post($this->api_endpoint . '/billing-portal', array(
            'body' => array(
                'license_key' => $license_key,
                'return_url' => admin_url('admin.php?page=yourbeans-settings'),
            ),
            'timeout' => 15,
            'sslverify' => true,
        ));

        if (is_wp_error($response)) {
            return false;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        return isset($data['url']) ? esc_url($data['url']) : false;
    }

    /**
     * Restrict plugin features if license inactive
     */
    public function maybe_restrict_features() {
        // Skip if no license key configured
        if (empty(get_option('yourbeans_license_key'))) {
            return;
        }

        if (!$this->is_license_active()) {
            add_action('admin_notices', array($this, 'license_inactive_notice'));

            // Disable product import
            remove_action('wp_ajax_yourbeans_import_product', array(yourbeans()->products, 'ajax_import_product'));

            // Add filter to block import attempts
            add_filter('yourbeans_can_import_products', '__return_false');
        }
    }

    /**
     * Display license inactive admin notice
     */
    public function license_inactive_notice() {
        $status = $this->get_license_status();
        $portal_url = $this->get_billing_portal_url();

        $manage_link = '';
        if ($portal_url) {
            $manage_link = sprintf(
                ' <a href="%s" target="_blank">%s</a>',
                esc_url($portal_url),
                __('Manage your subscription', 'yourbeans-coffee')
            );
        }
        ?>
        <div class="notice notice-error">
            <p>
                <strong><?php esc_html_e('YourBeans Coffee:', 'yourbeans-coffee'); ?></strong>
                <?php echo esc_html($status['message']); ?>
                <?php echo $manage_link; ?>
            </p>
        </div>
        <?php
    }

    /**
     * Mask license key for display
     *
     * @param string $license_key Full license key
     * @return string Masked license key (e.g., "YB-XXXX-XXXX-1234")
     */
    private function mask_license_key($license_key) {
        if (strlen($license_key) <= 8) {
            return str_repeat('X', strlen($license_key));
        }

        $visible_chars = 4;
        $masked = substr($license_key, 0, 3) . '-';
        $masked .= str_repeat('X', strlen($license_key) - 7) . '-';
        $masked .= substr($license_key, -$visible_chars);

        return $masked;
    }

    /**
     * AJAX handler for license activation
     */
    public function ajax_activate_license() {
        check_ajax_referer('yourbeans_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'yourbeans-coffee')));
        }

        $license_key = isset($_POST['license_key']) ? sanitize_text_field($_POST['license_key']) : '';
        $result = $this->activate_license($license_key);

        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }

    /**
     * AJAX handler for license deactivation
     */
    public function ajax_deactivate_license() {
        check_ajax_referer('yourbeans_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'yourbeans-coffee')));
        }

        $result = $this->deactivate_license();

        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }

    /**
     * AJAX handler for refreshing license status
     */
    public function ajax_refresh_license() {
        check_ajax_referer('yourbeans_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'yourbeans-coffee')));
        }

        delete_transient($this->cache_key);
        $status = $this->get_license_status();

        wp_send_json_success($status);
    }

    /**
     * Check if product import is allowed
     *
     * @return bool
     */
    public function can_import_products() {
        return apply_filters('yourbeans_can_import_products', $this->is_license_active());
    }
}
