<?php
/**
 * YourBeans Core Class
 *
 * Main plugin orchestration class - initializes all components
 *
 * @package YourBeans_Coffee
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class YourBeans_Core
 */
class YourBeans_Core {

    /**
     * Single instance
     *
     * @var YourBeans_Core
     */
    private static $instance = null;

    /**
     * API instance
     *
     * @var YourBeans_API
     */
    public $api;

    /**
     * Products instance
     *
     * @var YourBeans_Products
     */
    public $products;

    /**
     * Orders instance
     *
     * @var YourBeans_Orders
     */
    public $orders;

    /**
     * Stripe Connect instance
     *
     * @var YourBeans_Stripe_Connect
     */
    public $stripe;

    /**
     * Licensing instance
     *
     * @var YourBeans_Licensing
     */
    public $licensing;

    /**
     * Subscriptions instance
     *
     * @var YourBeans_Subscriptions
     */
    public $subscriptions;

    /**
     * Shipping instance
     *
     * @var YourBeans_Shipping
     */
    public $shipping;

    /**
     * Get single instance
     *
     * @return YourBeans_Core
     */
    public static function instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->includes();
        $this->init_components();
        $this->init_hooks();
    }

    /**
     * Include required files
     */
    private function includes() {
        // Core classes
        require_once YOURBEANS_PLUGIN_DIR . 'includes/class-yourbeans-api.php';
        require_once YOURBEANS_PLUGIN_DIR . 'includes/class-yourbeans-licensing.php';
        require_once YOURBEANS_PLUGIN_DIR . 'includes/class-yourbeans-shipping.php';
        require_once YOURBEANS_PLUGIN_DIR . 'includes/class-yourbeans-products.php';
        require_once YOURBEANS_PLUGIN_DIR . 'includes/class-yourbeans-subscriptions.php';
        require_once YOURBEANS_PLUGIN_DIR . 'includes/class-yourbeans-orders.php';
        require_once YOURBEANS_PLUGIN_DIR . 'includes/class-yourbeans-stripe-connect.php';
        require_once YOURBEANS_PLUGIN_DIR . 'includes/class-yourbeans-settings.php';

        // Admin classes
        if (is_admin()) {
            require_once YOURBEANS_PLUGIN_DIR . 'admin/class-yourbeans-admin.php';
        }
    }

    /**
     * Initialize components
     */
    private function init_components() {
        // Initialize core services
        $this->api = new YourBeans_API();
        $this->licensing = new YourBeans_Licensing();
        $this->shipping = new YourBeans_Shipping();
        $this->products = new YourBeans_Products($this->api);
        $this->subscriptions = new YourBeans_Subscriptions();
        $this->orders = new YourBeans_Orders($this->api);
        $this->stripe = new YourBeans_Stripe_Connect();

        // Initialize settings
        new YourBeans_Settings();

        // Initialize admin
        if (is_admin()) {
            new YourBeans_Admin();
        }
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Enqueue scripts
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));

        // Add settings link to plugins page
        add_filter('plugin_action_links_' . YOURBEANS_PLUGIN_BASENAME, array($this, 'add_settings_link'));

        // REST API endpoints
        add_action('rest_api_init', array($this, 'register_rest_routes'));

        // Cron for inventory sync
        add_action('yourbeans_sync_inventory', array($this, 'sync_inventory'));
        if (!wp_next_scheduled('yourbeans_sync_inventory')) {
            wp_schedule_event(time(), 'hourly', 'yourbeans_sync_inventory');
        }
    }

    /**
     * Enqueue admin scripts
     *
     * @param string $hook Current admin page hook
     */
    public function admin_scripts($hook) {
        // Only load on our pages
        if (strpos($hook, 'yourbeans') === false) {
            return;
        }

        wp_enqueue_style(
            'yourbeans-admin',
            YOURBEANS_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            YOURBEANS_VERSION
        );

        wp_enqueue_script(
            'yourbeans-admin',
            YOURBEANS_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            YOURBEANS_VERSION,
            true
        );

        wp_localize_script('yourbeans-admin', 'yourbeans', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('yourbeans_nonce'),
            'strings' => array(
                'importing' => __('Importing...', 'yourbeans-coffee'),
                'success' => __('Product imported successfully!', 'yourbeans-coffee'),
                'error' => __('Import failed. Please try again.', 'yourbeans-coffee'),
            ),
        ));
    }

    /**
     * Add settings link to plugins page
     *
     * @param array $links Existing links
     * @return array Modified links
     */
    public function add_settings_link($links) {
        $settings_link = sprintf(
            '<a href="%s">%s</a>',
            admin_url('admin.php?page=yourbeans-settings'),
            __('Settings', 'yourbeans-coffee')
        );
        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        // Fulfillment webhook endpoint
        register_rest_route('yourbeans/v1', '/webhook/fulfillment', array(
            'methods' => 'POST',
            'callback' => array($this->orders, 'handle_fulfillment_webhook'),
            'permission_callback' => array($this, 'verify_webhook_permission'),
        ));
    }

    /**
     * Verify webhook permission
     *
     * @param WP_REST_Request $request Request object
     * @return bool|WP_Error True if permitted, error otherwise
     */
    public function verify_webhook_permission($request) {
        $signature = $request->get_header('X-YourBeans-Signature');
        $webhook_secret = get_option('yourbeans_webhook_secret');

        if (!$webhook_secret || !$signature) {
            return new WP_Error('unauthorized', 'Invalid webhook signature', array('status' => 401));
        }

        $expected_signature = hash_hmac('sha256', $request->get_body(), $webhook_secret);

        if (!hash_equals($expected_signature, $signature)) {
            return new WP_Error('unauthorized', 'Invalid webhook signature', array('status' => 401));
        }

        return true;
    }

    /**
     * Sync inventory from MyShop.Coffee
     */
    public function sync_inventory() {
        if (!get_option('yourbeans_api_key')) {
            return;
        }

        try {
            $inventory = $this->api->get_inventory();

            if (empty($inventory)) {
                return;
            }

            foreach ($inventory as $item) {
                $this->products->update_inventory($item['myshop_product_id'], $item['stock']);
            }

            update_option('yourbeans_last_sync', current_time('mysql'));
            error_log('YourBeans inventory sync completed - ' . count($inventory) . ' products updated');

        } catch (Exception $e) {
            error_log('YourBeans inventory sync failed: ' . $e->getMessage());
        }
    }

    /**
     * Check if plugin is configured
     *
     * @return bool
     */
    public function is_configured() {
        return !empty(get_option('yourbeans_api_key')) && !empty(get_option('yourbeans_stripe_account_id'));
    }

    /**
     * Get plugin status
     *
     * @return array Status information
     */
    public function get_status() {
        $license_status = $this->licensing->get_license_status();

        return array(
            'license_active' => $license_status['is_active'],
            'license_status' => $license_status['status'],
            'api_connected' => !empty(get_option('yourbeans_api_key')),
            'stripe_connected' => !empty(get_option('yourbeans_stripe_account_id')),
            'last_sync' => get_option('yourbeans_last_sync', 'Never'),
            'products_imported' => $this->products->get_imported_count(),
        );
    }
}

/**
 * Helper function to get plugin instance
 *
 * @return YourBeans_Core
 */
function yourbeans() {
    return YourBeans_Core::instance();
}
