<?php
/**
 * YourBeans API Client
 *
 * Handles all communication with MyShop.Coffee API
 *
 * @package YourBeans_Coffee
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class YourBeans_API
 */
class YourBeans_API {

    /**
     * API base URL
     *
     * @var string
     */
    private $api_url = 'https://myshop.coffee/wp-json/yourbeans/v1';

    /**
     * API key
     *
     * @var string
     */
    private $api_key;

    /**
     * Constructor
     */
    public function __construct() {
        $this->api_key = get_option('yourbeans_api_key');
    }

    /**
     * Get available products from catalog
     *
     * @param array $args Query arguments
     * @return array Products list
     */
    public function get_products($args = array()) {
        $defaults = array(
            'per_page' => 50,
            'page' => 1,
            'category' => '',
            'search' => '',
        );

        $args = wp_parse_args($args, $defaults);

        return $this->request('GET', '/products', $args);
    }

    /**
     * Get single product details
     *
     * @param int $product_id MyShop product ID
     * @return array Product data
     */
    public function get_product($product_id) {
        return $this->request('GET', "/products/{$product_id}");
    }

    /**
     * Submit order for fulfillment
     *
     * @param array $order_data Order data
     * @return array Response with order ID
     */
    public function submit_order($order_data) {
        return $this->request('POST', '/orders', $order_data);
    }

    /**
     * Get order fulfillment status
     *
     * @param int $order_id MyShop order ID
     * @return array Order status data
     */
    public function get_order_status($order_id) {
        return $this->request('GET', "/orders/{$order_id}/status");
    }

    /**
     * Get current inventory levels
     *
     * @return array Inventory data
     */
    public function get_inventory() {
        return $this->request('GET', '/inventory');
    }

    /**
     * Get product categories
     *
     * @return array Categories list
     */
    public function get_categories() {
        return $this->request('GET', '/categories');
    }

    /**
     * Test API connection
     *
     * @return bool|WP_Error True if connection successful
     */
    public function test_connection() {
        try {
            $result = $this->request('GET', '/ping');
            return isset($result['status']) && $result['status'] === 'ok';
        } catch (Exception $e) {
            return new WP_Error('connection_failed', $e->getMessage());
        }
    }

    /**
     * Make API request
     *
     * @param string $method HTTP method
     * @param string $endpoint API endpoint
     * @param array  $data Request data
     * @return array Response data
     * @throws Exception On API error
     */
    private function request($method, $endpoint, $data = array()) {
        if (empty($this->api_key)) {
            throw new Exception(__('API key not configured', 'yourbeans-coffee'));
        }

        $url = $this->api_url . $endpoint;

        // Add query params for GET requests
        if ($method === 'GET' && !empty($data)) {
            $url = add_query_arg($data, $url);
        }

        $args = array(
            'method' => $method,
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'X-YourBeans-Version' => YOURBEANS_VERSION,
                'X-Store-URL' => home_url(),
            ),
            'timeout' => 30,
            'sslverify' => true,
        );

        // Add body for POST/PUT requests
        if (in_array($method, array('POST', 'PUT', 'PATCH')) && !empty($data)) {
            $args['body'] = wp_json_encode($data);
        }

        // Make request
        $response = wp_remote_request($url, $args);

        // Check for WordPress errors
        if (is_wp_error($response)) {
            error_log('YourBeans API Error: ' . $response->get_error_message());
            throw new Exception($response->get_error_message());
        }

        // Get response code
        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        // Log for debugging
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log(sprintf(
                'YourBeans API: %s %s - Response: %d',
                $method,
                $endpoint,
                $code
            ));
        }

        // Parse response
        $data = json_decode($body, true);

        // Handle errors
        if ($code >= 400) {
            $message = isset($data['message']) ? $data['message'] : __('API request failed', 'yourbeans-coffee');
            error_log('YourBeans API Error (' . $code . '): ' . $message);
            throw new Exception($message, $code);
        }

        return $data;
    }

    /**
     * Set API key (for testing)
     *
     * @param string $api_key API key
     */
    public function set_api_key($api_key) {
        $this->api_key = $api_key;
    }

    /**
     * Check if API is configured
     *
     * @return bool
     */
    public function is_configured() {
        return !empty($this->api_key);
    }
}
