/**
 * YourBeans Coffee - Admin JavaScript
 */

(function($) {
    'use strict';

    var YourBeans = {
        init: function() {
            this.bindEvents();
        },

        bindEvents: function() {
            // Load catalog button
            $('#yourbeans-load-catalog').on('click', this.loadCatalog.bind(this));

            // Search on enter
            $('#yourbeans-search').on('keypress', function(e) {
                if (e.which === 13) {
                    e.preventDefault();
                    YourBeans.loadCatalog();
                }
            });

            // Category filter change
            $('#yourbeans-category').on('change', this.loadCatalog.bind(this));

            // Import form submit
            $('#yourbeans-import-form').on('submit', this.importProduct.bind(this));

            // Modal close
            $('.yourbeans-modal-close').on('click', this.closeModal);
            $(document).on('click', '.yourbeans-modal', function(e) {
                if ($(e.target).hasClass('yourbeans-modal')) {
                    YourBeans.closeModal();
                }
            });

            // Disconnect Stripe
            $('.yourbeans-disconnect-stripe').on('click', this.disconnectStripe.bind(this));
        },

        loadCatalog: function() {
            var $grid = $('#yourbeans-catalog-grid');
            var $button = $('#yourbeans-load-catalog');

            $button.addClass('yourbeans-loading').prop('disabled', true);
            $grid.html('<p class="loading">Loading products...</p>');

            $.ajax({
                url: yourbeans.ajax_url,
                type: 'GET',
                data: {
                    action: 'yourbeans_get_catalog',
                    nonce: yourbeans.nonce,
                    search: $('#yourbeans-search').val(),
                    category: $('#yourbeans-category').val()
                },
                success: function(response) {
                    if (response.success && response.data.products) {
                        YourBeans.renderProducts(response.data.products);
                    } else {
                        $grid.html('<p class="error">' + (response.data.message || 'Failed to load products') + '</p>');
                    }
                },
                error: function() {
                    $grid.html('<p class="error">Failed to connect to server. Please try again.</p>');
                },
                complete: function() {
                    $button.removeClass('yourbeans-loading').prop('disabled', false);
                }
            });
        },

        renderProducts: function(products) {
            var $grid = $('#yourbeans-catalog-grid');

            if (!products || products.length === 0) {
                $grid.html('<p>No products found.</p>');
                return;
            }

            var html = '';

            products.forEach(function(product) {
                var importedClass = product.imported ? 'imported' : '';
                var importedBadge = product.imported ? '<span class="imported-badge">Already Imported</span>' : '';
                var buttonText = product.imported ? 'View Product' : 'Import';
                var buttonClass = product.imported ? 'button' : 'button button-primary';

                html += '<div class="yourbeans-product-card ' + importedClass + '" data-product-id="' + product.id + '">';
                html += '  <img src="' + (product.image || '') + '" alt="' + YourBeans.escapeHtml(product.title) + '">';
                html += '  <div class="product-info">';
                html += '    ' + importedBadge;
                html += '    <h3 class="product-title">' + YourBeans.escapeHtml(product.title) + '</h3>';
                html += '    <div class="product-price">From $' + parseFloat(product.base_price).toFixed(2) + '</div>';
                html += '    <div class="product-description">' + YourBeans.escapeHtml(product.short_description || '') + '</div>';
                html += '    <div class="product-actions">';

                if (product.imported) {
                    html += '      <a href="' + product.edit_url + '" class="' + buttonClass + '">' + buttonText + '</a>';
                } else {
                    html += '      <button type="button" class="' + buttonClass + ' yourbeans-import-btn" data-product-id="' + product.id + '" data-product-title="' + YourBeans.escapeHtml(product.title) + '">' + buttonText + '</button>';
                }

                html += '    </div>';
                html += '  </div>';
                html += '</div>';
            });

            $grid.html(html);

            // Bind import buttons
            $('.yourbeans-import-btn').on('click', function() {
                var productId = $(this).data('product-id');
                var productTitle = $(this).data('product-title');
                YourBeans.openImportModal(productId, productTitle);
            });
        },

        openImportModal: function(productId, productTitle) {
            $('#import-product-id').val(productId);
            $('#import-title').val('').attr('placeholder', productTitle);
            $('#yourbeans-import-modal').fadeIn(200);
        },

        closeModal: function() {
            $('.yourbeans-modal').fadeOut(200);
        },

        importProduct: function(e) {
            e.preventDefault();

            var $form = $('#yourbeans-import-form');
            var $button = $form.find('button[type="submit"]');

            $button.addClass('yourbeans-loading').prop('disabled', true);

            $.ajax({
                url: yourbeans.ajax_url,
                type: 'POST',
                data: {
                    action: 'yourbeans_import_product',
                    nonce: yourbeans.nonce,
                    product_id: $('#import-product-id').val(),
                    title: $('#import-title').val(),
                    markup: $('#import-markup').val()
                },
                success: function(response) {
                    if (response.success) {
                        alert(yourbeans.strings.success);
                        YourBeans.closeModal();

                        // Redirect to edit product
                        if (response.data.edit_url) {
                            window.location.href = response.data.edit_url;
                        } else {
                            // Reload catalog to show imported status
                            YourBeans.loadCatalog();
                        }
                    } else {
                        alert(response.data.message || yourbeans.strings.error);
                    }
                },
                error: function() {
                    alert(yourbeans.strings.error);
                },
                complete: function() {
                    $button.removeClass('yourbeans-loading').prop('disabled', false);
                }
            });
        },

        disconnectStripe: function(e) {
            e.preventDefault();

            if (!confirm('Are you sure you want to disconnect your Stripe account?')) {
                return;
            }

            var $button = $(e.target);
            $button.addClass('yourbeans-loading').prop('disabled', true);

            $.ajax({
                url: yourbeans.ajax_url,
                type: 'POST',
                data: {
                    action: 'yourbeans_disconnect_stripe',
                    nonce: yourbeans.nonce
                },
                success: function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert(response.data.message || 'Failed to disconnect');
                    }
                },
                error: function() {
                    alert('Failed to disconnect. Please try again.');
                },
                complete: function() {
                    $button.removeClass('yourbeans-loading').prop('disabled', false);
                }
            });
        },

        escapeHtml: function(text) {
            if (!text) return '';
            var map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return text.replace(/[&<>"']/g, function(m) { return map[m]; });
        }
    };

    $(document).ready(function() {
        YourBeans.init();
    });

})(jQuery);
