<?php
/**
 * YourBeans Admin
 *
 * Handles admin menu and pages
 *
 * @package YourBeans_Coffee
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class YourBeans_Admin
 */
class YourBeans_Admin {

    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_notices', array($this, 'admin_notices'));
    }

    /**
     * Add admin menu items
     */
    public function add_admin_menu() {
        // Main menu
        add_menu_page(
            __('YourBeans Coffee', 'yourbeans-coffee'),
            __('YourBeans', 'yourbeans-coffee'),
            'manage_woocommerce',
            'yourbeans',
            array($this, 'render_dashboard_page'),
            'dashicons-coffee',
            56
        );

        // Dashboard (same as main)
        add_submenu_page(
            'yourbeans',
            __('Dashboard', 'yourbeans-coffee'),
            __('Dashboard', 'yourbeans-coffee'),
            'manage_woocommerce',
            'yourbeans',
            array($this, 'render_dashboard_page')
        );

        // Products/Catalog
        add_submenu_page(
            'yourbeans',
            __('Product Catalog', 'yourbeans-coffee'),
            __('Import Products', 'yourbeans-coffee'),
            'manage_woocommerce',
            'yourbeans-products',
            array($this, 'render_products_page')
        );

        // Orders
        add_submenu_page(
            'yourbeans',
            __('YourBeans Orders', 'yourbeans-coffee'),
            __('Orders', 'yourbeans-coffee'),
            'manage_woocommerce',
            'yourbeans-orders',
            array($this, 'render_orders_page')
        );

        // Settings
        add_submenu_page(
            'yourbeans',
            __('Settings', 'yourbeans-coffee'),
            __('Settings', 'yourbeans-coffee'),
            'manage_woocommerce',
            'yourbeans-settings',
            array($this, 'render_settings_page')
        );
    }

    /**
     * Display admin notices
     */
    public function admin_notices() {
        // Show setup notice if not configured
        if (!get_option('yourbeans_api_key') && current_user_can('manage_woocommerce')) {
            $screen = get_current_screen();

            // Don't show on our settings page
            if ($screen && strpos($screen->id, 'yourbeans-settings') !== false) {
                return;
            }

            ?>
            <div class="notice notice-warning is-dismissible">
                <p>
                    <strong><?php esc_html_e('YourBeans Coffee', 'yourbeans-coffee'); ?>:</strong>
                    <?php
                    printf(
                        wp_kses(
                            __('Please <a href="%s">configure your API key</a> to start importing products.', 'yourbeans-coffee'),
                            array('a' => array('href' => array()))
                        ),
                        esc_url(admin_url('admin.php?page=yourbeans-settings'))
                    );
                    ?>
                </p>
            </div>
            <?php
        }

        // Show Stripe Connect notice if API configured but Stripe not connected
        if (get_option('yourbeans_api_key') && !get_option('yourbeans_stripe_account_id') && current_user_can('manage_woocommerce')) {
            $screen = get_current_screen();

            if ($screen && strpos($screen->id, 'yourbeans') !== false) {
                ?>
                <div class="notice notice-info is-dismissible">
                    <p>
                        <strong><?php esc_html_e('Stripe Connect Required', 'yourbeans-coffee'); ?>:</strong>
                        <?php
                        printf(
                            wp_kses(
                                __('Connect your Stripe account in <a href="%s">Settings</a> to enable automatic payment splitting.', 'yourbeans-coffee'),
                                array('a' => array('href' => array()))
                            ),
                            esc_url(admin_url('admin.php?page=yourbeans-settings'))
                        );
                        ?>
                    </p>
                </div>
                <?php
            }
        }
    }

    /**
     * Render dashboard page
     */
    public function render_dashboard_page() {
        $status = yourbeans()->get_status();
        $license_status = yourbeans()->licensing->get_license_status();
        ?>
        <div class="wrap yourbeans-dashboard">
            <h1><?php esc_html_e('YourBeans Coffee Dashboard', 'yourbeans-coffee'); ?></h1>

            <div class="yourbeans-status-cards">
                <!-- License Status -->
                <div class="yourbeans-card">
                    <h2><?php esc_html_e('Subscription Status', 'yourbeans-coffee'); ?></h2>
                    <table class="widefat">
                        <tr>
                            <th><?php esc_html_e('Plugin License', 'yourbeans-coffee'); ?></th>
                            <td>
                                <?php if ($license_status['is_active']): ?>
                                    <span class="status-badge status-connected"><?php esc_html_e('Active', 'yourbeans-coffee'); ?></span>
                                <?php elseif ($license_status['status'] === 'not_configured'): ?>
                                    <span class="status-badge status-pending"><?php esc_html_e('Not Configured', 'yourbeans-coffee'); ?></span>
                                <?php else: ?>
                                    <span class="status-badge status-disconnected"><?php echo esc_html(ucfirst($license_status['status'])); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php if (!empty($license_status['license_key'])): ?>
                        <tr>
                            <th><?php esc_html_e('License Key', 'yourbeans-coffee'); ?></th>
                            <td><code><?php echo esc_html($license_status['license_key']); ?></code></td>
                        </tr>
                        <?php endif; ?>
                    </table>
                    <?php
                    $portal_url = yourbeans()->licensing->get_billing_portal_url();
                    if ($portal_url):
                    ?>
                    <p style="margin-top: 10px;">
                        <a href="<?php echo esc_url($portal_url); ?>" class="button" target="_blank">
                            <?php esc_html_e('Manage Subscription', 'yourbeans-coffee'); ?>
                        </a>
                    </p>
                    <?php endif; ?>
                </div>

                <!-- Connection Status -->
                <div class="yourbeans-card">
                    <h2><?php esc_html_e('Connection Status', 'yourbeans-coffee'); ?></h2>
                    <table class="widefat">
                        <tr>
                            <th><?php esc_html_e('MyShop.Coffee API', 'yourbeans-coffee'); ?></th>
                            <td>
                                <?php if ($status['api_connected']): ?>
                                    <span class="status-badge status-connected"><?php esc_html_e('Connected', 'yourbeans-coffee'); ?></span>
                                <?php else: ?>
                                    <span class="status-badge status-disconnected"><?php esc_html_e('Not Connected', 'yourbeans-coffee'); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e('Stripe Connect', 'yourbeans-coffee'); ?></th>
                            <td>
                                <?php if ($status['stripe_connected']): ?>
                                    <span class="status-badge status-connected"><?php esc_html_e('Connected', 'yourbeans-coffee'); ?></span>
                                <?php else: ?>
                                    <span class="status-badge status-disconnected"><?php esc_html_e('Not Connected', 'yourbeans-coffee'); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e('Last Inventory Sync', 'yourbeans-coffee'); ?></th>
                            <td><?php echo esc_html($status['last_sync'] ?: __('Never', 'yourbeans-coffee')); ?></td>
                        </tr>
                    </table>
                </div>

                <!-- Quick Stats -->
                <div class="yourbeans-card">
                    <h2><?php esc_html_e('Quick Stats', 'yourbeans-coffee'); ?></h2>
                    <div class="yourbeans-stats">
                        <div class="stat">
                            <span class="stat-number"><?php echo esc_html($status['products_imported']); ?></span>
                            <span class="stat-label"><?php esc_html_e('Products Imported', 'yourbeans-coffee'); ?></span>
                        </div>
                    </div>
                </div>

                <!-- Quick Actions -->
                <div class="yourbeans-card">
                    <h2><?php esc_html_e('Quick Actions', 'yourbeans-coffee'); ?></h2>
                    <p>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=yourbeans-products')); ?>" class="button button-primary">
                            <?php esc_html_e('Import Products', 'yourbeans-coffee'); ?>
                        </a>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=yourbeans-settings')); ?>" class="button">
                            <?php esc_html_e('Settings', 'yourbeans-coffee'); ?>
                        </a>
                    </p>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render products/catalog page
     */
    public function render_products_page() {
        if (!get_option('yourbeans_api_key')) {
            ?>
            <div class="wrap">
                <h1><?php esc_html_e('Import Products', 'yourbeans-coffee'); ?></h1>
                <div class="notice notice-warning">
                    <p>
                        <?php
                        printf(
                            wp_kses(
                                __('Please <a href="%s">configure your API key</a> first.', 'yourbeans-coffee'),
                                array('a' => array('href' => array()))
                            ),
                            esc_url(admin_url('admin.php?page=yourbeans-settings'))
                        );
                        ?>
                    </p>
                </div>
            </div>
            <?php
            return;
        }
        ?>
        <div class="wrap yourbeans-products">
            <h1><?php esc_html_e('Import Products from YourBeans Catalog', 'yourbeans-coffee'); ?></h1>

            <div class="yourbeans-catalog-filters">
                <input type="text" id="yourbeans-search" placeholder="<?php esc_attr_e('Search products...', 'yourbeans-coffee'); ?>" />
                <select id="yourbeans-category">
                    <option value=""><?php esc_html_e('All Categories', 'yourbeans-coffee'); ?></option>
                </select>
                <button type="button" class="button" id="yourbeans-load-catalog">
                    <?php esc_html_e('Load Catalog', 'yourbeans-coffee'); ?>
                </button>
            </div>

            <div id="yourbeans-catalog-grid" class="yourbeans-product-grid">
                <p class="loading"><?php esc_html_e('Click "Load Catalog" to browse available products.', 'yourbeans-coffee'); ?></p>
            </div>

            <!-- Import Modal -->
            <div id="yourbeans-import-modal" class="yourbeans-modal" style="display:none;">
                <div class="yourbeans-modal-content">
                    <span class="yourbeans-modal-close">&times;</span>
                    <h2><?php esc_html_e('Import Product', 'yourbeans-coffee'); ?></h2>
                    <form id="yourbeans-import-form">
                        <input type="hidden" id="import-product-id" name="product_id" />

                        <p>
                            <label for="import-title"><?php esc_html_e('Product Title', 'yourbeans-coffee'); ?></label>
                            <input type="text" id="import-title" name="title" class="widefat" />
                            <span class="description"><?php esc_html_e('Leave blank to use original title', 'yourbeans-coffee'); ?></span>
                        </p>

                        <p>
                            <label for="import-markup"><?php esc_html_e('Markup %', 'yourbeans-coffee'); ?></label>
                            <input type="number" id="import-markup" name="markup"
                                   value="<?php echo esc_attr(get_option('yourbeans_default_markup', 50)); ?>"
                                   min="0" max="500" step="5" />
                        </p>

                        <p class="submit">
                            <button type="submit" class="button button-primary">
                                <?php esc_html_e('Import Product', 'yourbeans-coffee'); ?>
                            </button>
                        </p>
                    </form>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render orders page
     */
    public function render_orders_page() {
        ?>
        <div class="wrap yourbeans-orders">
            <h1><?php esc_html_e('YourBeans Orders', 'yourbeans-coffee'); ?></h1>

            <p><?php esc_html_e('Orders containing YourBeans products are automatically submitted for fulfillment when payment is received.', 'yourbeans-coffee'); ?></p>

            <?php
            // Query orders with YourBeans products
            $args = array(
                'limit' => 20,
                'orderby' => 'date',
                'order' => 'DESC',
                'meta_key' => '_yourbeans_submitted',
            );

            $orders = wc_get_orders($args);

            if (empty($orders)) {
                echo '<p>' . esc_html__('No YourBeans orders found.', 'yourbeans-coffee') . '</p>';
            } else {
                ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Order', 'yourbeans-coffee'); ?></th>
                            <th><?php esc_html_e('Date', 'yourbeans-coffee'); ?></th>
                            <th><?php esc_html_e('Status', 'yourbeans-coffee'); ?></th>
                            <th><?php esc_html_e('MyShop ID', 'yourbeans-coffee'); ?></th>
                            <th><?php esc_html_e('Tracking', 'yourbeans-coffee'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($orders as $order): ?>
                            <tr>
                                <td>
                                    <a href="<?php echo esc_url($order->get_edit_order_url()); ?>">
                                        #<?php echo esc_html($order->get_order_number()); ?>
                                    </a>
                                </td>
                                <td><?php echo esc_html($order->get_date_created()->date_i18n(get_option('date_format'))); ?></td>
                                <td><?php echo esc_html(wc_get_order_status_name($order->get_status())); ?></td>
                                <td><?php echo esc_html($order->get_meta('_yourbeans_order_id') ?: '-'); ?></td>
                                <td><?php echo esc_html($order->get_meta('_yourbeans_tracking_number') ?: '-'); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                <?php
            }
            ?>
        </div>
        <?php
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        $stripe = yourbeans()->stripe;
        $license_status = yourbeans()->licensing->get_license_status();
        ?>
        <div class="wrap yourbeans-settings">
            <h1><?php esc_html_e('YourBeans Coffee Settings', 'yourbeans-coffee'); ?></h1>

            <?php settings_errors('yourbeans_messages'); ?>

            <!-- License Status Card -->
            <div class="yourbeans-card">
                <h2><?php esc_html_e('Plugin License', 'yourbeans-coffee'); ?></h2>

                <table class="form-table">
                    <tr>
                        <th><?php esc_html_e('Subscription Status', 'yourbeans-coffee'); ?></th>
                        <td>
                            <?php if ($license_status['is_active']): ?>
                                <span class="status-badge status-connected"><?php esc_html_e('Active', 'yourbeans-coffee'); ?></span>
                                <?php
                                $portal_url = yourbeans()->licensing->get_billing_portal_url();
                                if ($portal_url):
                                ?>
                                <br><br>
                                <a href="<?php echo esc_url($portal_url); ?>" class="button" target="_blank">
                                    <?php esc_html_e('Manage Subscription', 'yourbeans-coffee'); ?>
                                </a>
                                <?php endif; ?>
                            <?php elseif ($license_status['status'] === 'not_configured'): ?>
                                <span class="status-badge status-pending"><?php esc_html_e('Not Configured', 'yourbeans-coffee'); ?></span>
                            <?php else: ?>
                                <span class="status-badge status-disconnected"><?php echo esc_html(ucfirst($license_status['status'])); ?></span>
                                <br><small><?php echo esc_html($license_status['message']); ?></small>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="yourbeans_license_key"><?php esc_html_e('License Key', 'yourbeans-coffee'); ?></label></th>
                        <td>
                            <input type="text"
                                   id="yourbeans_license_key"
                                   name="yourbeans_license_key_input"
                                   value="<?php echo esc_attr(get_option('yourbeans_license_key', '')); ?>"
                                   class="regular-text"
                                   placeholder="YB-XXXX-XXXX-XXXX" />
                            <button type="button" class="button" id="yourbeans-activate-license">
                                <?php esc_html_e('Activate', 'yourbeans-coffee'); ?>
                            </button>
                            <?php if (!empty(get_option('yourbeans_license_key'))): ?>
                            <button type="button" class="button" id="yourbeans-deactivate-license">
                                <?php esc_html_e('Deactivate', 'yourbeans-coffee'); ?>
                            </button>
                            <?php endif; ?>
                            <p class="description">
                                <?php esc_html_e('Enter your YourBeans license key from your MyShop.Coffee vendor dashboard.', 'yourbeans-coffee'); ?>
                            </p>
                        </td>
                    </tr>
                </table>
            </div>

            <!-- Connection Status Card -->
            <div class="yourbeans-card">
                <h2><?php esc_html_e('Connection Status', 'yourbeans-coffee'); ?></h2>

                <table class="form-table">
                    <tr>
                        <th><?php esc_html_e('MyShop.Coffee API', 'yourbeans-coffee'); ?></th>
                        <td>
                            <?php if (get_option('yourbeans_api_key')): ?>
                                <span class="status-badge status-connected"><?php esc_html_e('Connected', 'yourbeans-coffee'); ?></span>
                            <?php else: ?>
                                <span class="status-badge status-disconnected"><?php esc_html_e('Not Connected', 'yourbeans-coffee'); ?></span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e('Stripe Connect', 'yourbeans-coffee'); ?></th>
                        <td>
                            <?php if ($stripe->is_connected()): ?>
                                <span class="status-badge status-connected"><?php esc_html_e('Connected', 'yourbeans-coffee'); ?></span>
                                <br>
                                <small><?php echo esc_html($stripe->get_account_id()); ?></small>
                                <br>
                                <button type="button" class="button button-small yourbeans-disconnect-stripe">
                                    <?php esc_html_e('Disconnect', 'yourbeans-coffee'); ?>
                                </button>
                            <?php else: ?>
                                <span class="status-badge status-disconnected"><?php esc_html_e('Not Connected', 'yourbeans-coffee'); ?></span>
                                <?php
                                $connect_url = $stripe->get_connect_url();
                                if ($connect_url):
                                ?>
                                    <br>
                                    <a href="<?php echo esc_url($connect_url); ?>" class="button button-primary">
                                        <?php esc_html_e('Connect Stripe Account', 'yourbeans-coffee'); ?>
                                    </a>
                                <?php else: ?>
                                    <br>
                                    <small class="description">
                                        <?php esc_html_e('Stripe Connect Client ID not configured.', 'yourbeans-coffee'); ?>
                                    </small>
                                <?php endif; ?>
                            <?php endif; ?>
                        </td>
                    </tr>
                </table>
            </div>

            <!-- Shipping Rates Card (Read-only) -->
            <div class="yourbeans-card">
                <h2><?php esc_html_e('Shipping Rates', 'yourbeans-coffee'); ?></h2>
                <p class="description" style="margin-bottom: 15px;">
                    <?php esc_html_e('Shipping rates are set by MyShop.Coffee and are used to calculate the platform fee on each order.', 'yourbeans-coffee'); ?>
                </p>
                <?php echo yourbeans()->shipping->get_rates_table_html(); ?>
            </div>

            <!-- Settings Form -->
            <div class="yourbeans-card">
                <h2><?php esc_html_e('Default Settings', 'yourbeans-coffee'); ?></h2>
                <form method="post" action="options.php">
                    <?php
                    settings_fields('yourbeans_settings');
                    do_settings_sections('yourbeans_settings');
                    submit_button();
                    ?>
                </form>
            </div>

            <!-- Payment Split Info -->
            <div class="yourbeans-card">
                <h2><?php esc_html_e('How Payment Splitting Works', 'yourbeans-coffee'); ?></h2>
                <p><?php esc_html_e('When a customer purchases a YourBeans product, the payment is automatically split:', 'yourbeans-coffee'); ?></p>

                <div style="background: #f0f0f0; padding: 15px; border-radius: 4px; margin: 15px 0;">
                    <code style="display: block; margin-bottom: 10px;">
                        <?php esc_html_e('Platform Fee = COGS (Cost of Goods Sold) + Shipping Cost', 'yourbeans-coffee'); ?>
                    </code>
                    <code style="display: block;">
                        <?php esc_html_e('Your Margin = Retail Price - Platform Fee', 'yourbeans-coffee'); ?>
                    </code>
                </div>

                <table class="widefat" style="max-width: 500px;">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Example Order', 'yourbeans-coffee'); ?></th>
                            <th style="text-align: right;"><?php esc_html_e('Amount', 'yourbeans-coffee'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><?php esc_html_e('Customer Pays (1kg Premium Blend)', 'yourbeans-coffee'); ?></td>
                            <td style="text-align: right;">$55.00</td>
                        </tr>
                        <tr>
                            <td><?php esc_html_e('+ Shipping', 'yourbeans-coffee'); ?></td>
                            <td style="text-align: right;">$10.00</td>
                        </tr>
                        <tr style="background: #e8f4fc;">
                            <td><strong><?php esc_html_e('Total Order', 'yourbeans-coffee'); ?></strong></td>
                            <td style="text-align: right;"><strong>$65.00</strong></td>
                        </tr>
                        <tr>
                            <td><?php esc_html_e('COGS (to MyShop)', 'yourbeans-coffee'); ?></td>
                            <td style="text-align: right;">-$28.00</td>
                        </tr>
                        <tr>
                            <td><?php esc_html_e('Shipping Cost (to MyShop)', 'yourbeans-coffee'); ?></td>
                            <td style="text-align: right;">-$8.50</td>
                        </tr>
                        <tr style="background: #d4edda;">
                            <td><strong><?php esc_html_e('Your Margin', 'yourbeans-coffee'); ?></strong></td>
                            <td style="text-align: right;"><strong>$28.50</strong></td>
                        </tr>
                    </tbody>
                </table>

                <p class="description" style="margin-top: 15px;">
                    <?php esc_html_e('COGS is set per product variant and cannot be modified. You control your retail prices by setting your markup percentage.', 'yourbeans-coffee'); ?>
                </p>
            </div>
        </div>

        <script>
        jQuery(function($) {
            // License activation
            $('#yourbeans-activate-license').on('click', function() {
                var $button = $(this);
                var licenseKey = $('#yourbeans_license_key').val();

                if (!licenseKey) {
                    alert('<?php echo esc_js(__('Please enter a license key.', 'yourbeans-coffee')); ?>');
                    return;
                }

                $button.prop('disabled', true).text('<?php echo esc_js(__('Activating...', 'yourbeans-coffee')); ?>');

                $.ajax({
                    url: yourbeans.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'yourbeans_activate_license',
                        nonce: yourbeans.nonce,
                        license_key: licenseKey
                    },
                    success: function(response) {
                        if (response.success) {
                            alert(response.data.message);
                            location.reload();
                        } else {
                            alert(response.data.message || '<?php echo esc_js(__('Activation failed.', 'yourbeans-coffee')); ?>');
                        }
                    },
                    error: function() {
                        alert('<?php echo esc_js(__('Connection error. Please try again.', 'yourbeans-coffee')); ?>');
                    },
                    complete: function() {
                        $button.prop('disabled', false).text('<?php echo esc_js(__('Activate', 'yourbeans-coffee')); ?>');
                    }
                });
            });

            // License deactivation
            $('#yourbeans-deactivate-license').on('click', function() {
                if (!confirm('<?php echo esc_js(__('Are you sure you want to deactivate your license?', 'yourbeans-coffee')); ?>')) {
                    return;
                }

                var $button = $(this);
                $button.prop('disabled', true);

                $.ajax({
                    url: yourbeans.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'yourbeans_deactivate_license',
                        nonce: yourbeans.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            alert(response.data.message);
                            location.reload();
                        } else {
                            alert(response.data.message || '<?php echo esc_js(__('Deactivation failed.', 'yourbeans-coffee')); ?>');
                        }
                    },
                    error: function() {
                        alert('<?php echo esc_js(__('Connection error. Please try again.', 'yourbeans-coffee')); ?>');
                    },
                    complete: function() {
                        $button.prop('disabled', false);
                    }
                });
            });
        });
        </script>
        <?php
    }
}
