# YourBeans Coffee - WordPress/WooCommerce Plugin

**Professional coffee dropshipping integration for WooCommerce stores**
*By MyShop Enterprises Pty Ltd*

---

## Overview

This plugin connects external WordPress/WooCommerce stores to the MyShop.Coffee platform, enabling store owners to import premium coffee products, customize them with their own branding, and sell them (including subscriptions) with automated fulfillment.

---

## Business Model: Dual Subscription

```
┌─────────────────────────────────────────────────────────────────┐
│              SUBSCRIPTION LAYER 1: PLUGIN ACCESS                │
│                                                                 │
│     Store Owner subscribes via Stripe → MyShop Enterprises      │
│                      $X/month for plugin license                │
│                                                                 │
│     • Managed directly by MyShop via Stripe Billing             │
│     • Plugin validates subscription status on activation        │
│     • Grace period for failed payments                          │
└─────────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────────┐
│            SUBSCRIPTION LAYER 2: CUSTOMER SUBSCRIPTIONS          │
│                                                                 │
│     End Customer subscribes → Store Owner's WooCommerce         │
│              Weekly / Fortnightly / Monthly coffee              │
│                              │                                  │
│                              ▼                                  │
│              Stripe Connect splits each recurring payment       │
│              MyShop fulfills each recurring order               │
└─────────────────────────────────────────────────────────────────┘
```

---

## Payment Split Calculation

### Formula: Platform Fee = COGS + Shipping

```
Customer Order: $65.00
├── Product: Premium Blend 1kg × 1 = $55.00
│   └── COGS (from MyShop): $28.00
├── Shipping: $10.00
│   └── Shipping Cost (to MyShop): $8.50
└── Total Platform Fee: $28.00 + $8.50 = $36.50

Stripe Connect Split:
├── MyShop Receives: $36.50 (COGS + Shipping)
└── Store Owner Keeps: $28.50 (Margin)
```

### Per-Variant COGS Configuration

COGS is set **per variant** (not per product) because costs differ by size:

| Variant | Size | COGS | Typical Retail | Margin |
|---------|------|------|----------------|--------|
| Premium Blend - 250g | 250g | $8.50 | $18.00 | $9.50 |
| Premium Blend - 1kg | 1kg | $28.00 | $55.00 | $27.00 |

### Shipping Cost Configuration

Shipping costs pulled from MyShop's shipping matrix:

| Zone | Base Rate | Per-kg Rate |
|------|-----------|-------------|
| Metro (NSW/VIC/QLD) | $8.50 | +$2.00/kg |
| Regional | $10.50 | +$2.50/kg |
| Remote (WA/NT/TAS) | $14.50 | +$3.00/kg |

---

## Product Customization Rules

### What Store Owners CAN Customize (Branding)

| Field | Customizable | Notes |
|-------|--------------|-------|
| **Product Title** | ✅ Yes | e.g., "Joe's Premium Roast" |
| **Description** | ✅ Yes | Full HTML supported |
| **Short Description** | ✅ Yes | For catalog display |
| **Featured Image** | ✅ Yes | Upload own branded images |
| **Gallery Images** | ✅ Yes | Multiple images supported |
| **Retail Price** | ✅ Yes | Set via markup % |
| **Product Categories** | ✅ Yes | Organize in their store |
| **Tags** | ✅ Yes | SEO and filtering |

### What Store Owners CANNOT Change (Core Attributes)

| Attribute | Fixed | Why |
|-----------|-------|-----|
| **Grind Options** | ✅ Fixed | Fulfillment requires exact grind types |
| **Size/Weight Options** | ✅ Fixed | COGS and shipping depend on weight |
| **SKU Structure** | ✅ Fixed | Links to MyShop inventory system |
| **Core Product ID** | ✅ Fixed | Required for order fulfillment |
| **Available Variants** | ✅ Fixed | 12 variants (2 sizes × 6 grinds) |

### Fixed Variant Matrix

Every imported product has exactly 12 variants:

```
Sizes: 250g, 1kg
Grinds: Whole Bean, Espresso, Filter, Plunger, Stovetop, Aeropress

Resulting Variants:
├── 250g - Whole Bean
├── 250g - Espresso
├── 250g - Filter
├── 250g - Plunger
├── 250g - Stovetop
├── 250g - Aeropress
├── 1kg - Whole Bean
├── 1kg - Espresso
├── 1kg - Filter
├── 1kg - Plunger
├── 1kg - Stovetop
└── 1kg - Aeropress
```

---

## Plugin Architecture

### Directory Structure

```
yourbeans-coffee/
├── yourbeans-coffee.php              # Main plugin file
├── readme.txt                        # WordPress readme
├── assets/
│   ├── css/
│   │   └── admin.css                # Admin styles
│   └── js/
│       └── admin.js                 # Admin JavaScript
├── includes/
│   ├── class-yourbeans-core.php              # Core orchestration
│   ├── class-yourbeans-api.php               # MyShop.Coffee API client
│   ├── class-yourbeans-licensing.php         # Stripe subscription validation
│   ├── class-yourbeans-products.php          # Product import/customization
│   ├── class-yourbeans-subscriptions.php     # Customer subscription functionality
│   ├── class-yourbeans-orders.php            # Order handling
│   ├── class-yourbeans-stripe-connect.php    # Payment splitting
│   ├── class-yourbeans-shipping.php          # Shipping cost calculation
│   └── class-yourbeans-settings.php          # Settings management
├── admin/
│   ├── class-yourbeans-admin.php             # Admin pages
│   └── views/
│       ├── dashboard.php                     # Main dashboard
│       ├── products.php                      # Product catalog/import
│       ├── product-edit.php                  # Product customization
│       ├── cogs-settings.php                 # COGS configuration
│       └── settings.php                      # General settings
├── templates/
│   ├── subscription-options.php              # Frontend subscription display
│   └── emails/                               # Email templates
└── languages/
    └── yourbeans-coffee.pot                  # Translations
```

---

## Core Classes

### 1. Plugin Licensing (`class-yourbeans-licensing.php`)

Validates store owner's subscription to the plugin (via MyShop's Stripe).

```php
<?php
class YourBeans_Licensing {

    private $api_endpoint = 'https://myshop.coffee/wp-json/yourbeans/v1/license';

    /**
     * Check if store has active plugin subscription
     */
    public function is_license_active() {
        $cached = get_transient('yourbeans_license_status');
        if ($cached !== false) {
            return $cached === 'active';
        }

        $status = $this->verify_license();
        set_transient('yourbeans_license_status', $status, HOUR_IN_SECONDS);

        return $status === 'active';
    }

    /**
     * Verify license with MyShop API
     */
    private function verify_license() {
        $license_key = get_option('yourbeans_license_key');
        $store_url = home_url();

        $response = wp_remote_post($this->api_endpoint . '/verify', array(
            'body' => array(
                'license_key' => $license_key,
                'store_url' => $store_url,
            ),
            'timeout' => 15,
        ));

        if (is_wp_error($response)) {
            return 'error';
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        return $data['status'] ?? 'invalid';
    }

    /**
     * Get subscription management URL (Stripe Customer Portal)
     */
    public function get_billing_portal_url() {
        $response = wp_remote_post($this->api_endpoint . '/billing-portal', array(
            'body' => array(
                'license_key' => get_option('yourbeans_license_key'),
                'return_url' => admin_url('admin.php?page=yourbeans-settings'),
            ),
        ));

        $data = json_decode(wp_remote_retrieve_body($response), true);
        return $data['url'] ?? '';
    }

    /**
     * Restrict plugin features if license inactive
     */
    public function maybe_restrict_features() {
        if (!$this->is_license_active()) {
            add_action('admin_notices', array($this, 'license_inactive_notice'));
            // Disable product import, but allow existing products to function
            remove_action('wp_ajax_yourbeans_import_product', array(yourbeans()->products, 'ajax_import_product'));
        }
    }

    /**
     * License inactive admin notice
     */
    public function license_inactive_notice() {
        $portal_url = $this->get_billing_portal_url();
        ?>
        <div class="notice notice-error">
            <p>
                <strong>YourBeans Coffee:</strong>
                Your subscription is inactive.
                <a href="<?php echo esc_url($portal_url); ?>">Manage your subscription</a>
                to continue importing products.
            </p>
        </div>
        <?php
    }
}
```

### 2. Customer Subscriptions (`class-yourbeans-subscriptions.php`)

Built-in subscription functionality for store owners to sell coffee subscriptions.

```php
<?php
class YourBeans_Subscriptions {

    /**
     * Subscription intervals
     */
    private $intervals = array(
        'weekly' => array(
            'label' => 'Weekly',
            'days' => 7,
        ),
        'fortnightly' => array(
            'label' => 'Every 2 Weeks',
            'days' => 14,
        ),
        'monthly' => array(
            'label' => 'Monthly',
            'days' => 30,
        ),
    );

    public function __construct() {
        // Add subscription options to YourBeans products
        add_action('woocommerce_before_add_to_cart_button', array($this, 'display_subscription_options'));
        add_filter('woocommerce_add_cart_item_data', array($this, 'add_subscription_to_cart'), 10, 3);
        add_filter('woocommerce_get_item_data', array($this, 'display_subscription_in_cart'), 10, 2);

        // Handle subscription orders
        add_action('woocommerce_checkout_create_order_line_item', array($this, 'save_subscription_to_order'), 10, 4);
        add_action('woocommerce_order_status_completed', array($this, 'schedule_renewal'));

        // Cron for renewals
        add_action('yourbeans_process_subscription_renewal', array($this, 'process_renewal'), 10, 2);

        // Customer subscription management
        add_action('woocommerce_account_yourbeans-subscriptions_endpoint', array($this, 'subscription_management_page'));
    }

    /**
     * Display subscription options on product page
     */
    public function display_subscription_options() {
        global $product;

        if (!$this->is_yourbeans_product($product)) {
            return;
        }

        $subscription_enabled = get_post_meta($product->get_id(), '_yourbeans_subscription_enabled', true);

        if ($subscription_enabled !== 'yes') {
            return;
        }

        $discount = get_post_meta($product->get_id(), '_yourbeans_subscription_discount', true) ?: 10;
        ?>
        <div class="yourbeans-subscription-options">
            <h4>Purchase Options</h4>

            <label class="subscription-option">
                <input type="radio" name="yourbeans_purchase_type" value="one-time" checked>
                <span>One-time purchase</span>
            </label>

            <label class="subscription-option subscription-choice">
                <input type="radio" name="yourbeans_purchase_type" value="subscription">
                <span>Subscribe & Save <?php echo esc_html($discount); ?>%</span>
            </label>

            <div class="subscription-interval" style="display:none;">
                <label for="yourbeans_interval">Delivery Frequency:</label>
                <select name="yourbeans_interval" id="yourbeans_interval">
                    <?php foreach ($this->intervals as $key => $interval): ?>
                        <option value="<?php echo esc_attr($key); ?>">
                            <?php echo esc_html($interval['label']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
        </div>

        <script>
        jQuery(function($) {
            $('input[name="yourbeans_purchase_type"]').on('change', function() {
                if ($(this).val() === 'subscription') {
                    $('.subscription-interval').slideDown();
                } else {
                    $('.subscription-interval').slideUp();
                }
            });
        });
        </script>
        <?php
    }

    /**
     * Add subscription data to cart
     */
    public function add_subscription_to_cart($cart_item_data, $product_id, $variation_id) {
        if (isset($_POST['yourbeans_purchase_type']) && $_POST['yourbeans_purchase_type'] === 'subscription') {
            $cart_item_data['yourbeans_subscription'] = array(
                'type' => 'subscription',
                'interval' => sanitize_text_field($_POST['yourbeans_interval'] ?? 'monthly'),
            );

            // Apply discount
            $discount = get_post_meta($product_id, '_yourbeans_subscription_discount', true) ?: 10;
            $cart_item_data['yourbeans_subscription']['discount'] = $discount;
        }

        return $cart_item_data;
    }

    /**
     * Schedule next renewal after order completion
     */
    public function schedule_renewal($order_id) {
        $order = wc_get_order($order_id);

        foreach ($order->get_items() as $item) {
            $subscription_data = $item->get_meta('_yourbeans_subscription');

            if (!$subscription_data || $subscription_data['type'] !== 'subscription') {
                continue;
            }

            $interval = $subscription_data['interval'];
            $days = $this->intervals[$interval]['days'] ?? 30;
            $next_renewal = strtotime("+{$days} days");

            // Create subscription record
            $subscription_id = $this->create_subscription_record($order, $item, $subscription_data);

            // Schedule renewal
            wp_schedule_single_event($next_renewal, 'yourbeans_process_subscription_renewal', array(
                $subscription_id,
                $order->get_customer_id(),
            ));
        }
    }

    /**
     * Process subscription renewal
     */
    public function process_renewal($subscription_id, $customer_id) {
        $subscription = $this->get_subscription($subscription_id);

        if (!$subscription || $subscription['status'] !== 'active') {
            return;
        }

        // Create renewal order
        $renewal_order = wc_create_order(array(
            'customer_id' => $customer_id,
            'status' => 'pending',
        ));

        // Copy items from original
        $original_order = wc_get_order($subscription['original_order_id']);

        foreach ($original_order->get_items() as $item) {
            if ($item->get_meta('_yourbeans_subscription')) {
                $renewal_order->add_product(
                    wc_get_product($item->get_variation_id() ?: $item->get_product_id()),
                    $item->get_quantity()
                );
            }
        }

        // Copy addresses
        $renewal_order->set_address($original_order->get_address('billing'), 'billing');
        $renewal_order->set_address($original_order->get_address('shipping'), 'shipping');

        // Apply subscription discount
        $discount = $subscription['discount'] ?? 10;
        // ... apply discount logic

        $renewal_order->calculate_totals();
        $renewal_order->save();

        // Charge customer's saved payment method
        $this->charge_renewal($renewal_order, $customer_id);

        // Schedule next renewal
        $days = $this->intervals[$subscription['interval']]['days'] ?? 30;
        wp_schedule_single_event(strtotime("+{$days} days"), 'yourbeans_process_subscription_renewal', array(
            $subscription_id,
            $customer_id,
        ));
    }

    /**
     * Check if product is YourBeans product
     */
    private function is_yourbeans_product($product) {
        return !empty(get_post_meta($product->get_id(), '_yourbeans_myshop_product_id', true));
    }
}
```

### 3. Products with Customization Rules (`class-yourbeans-products.php`)

Updated to handle customizable vs fixed fields.

```php
<?php
class YourBeans_Products {

    /**
     * Fixed attributes - CANNOT be changed by store owner
     */
    private $fixed_sizes = array('250g', '1kg');
    private $fixed_grinds = array(
        'Whole Bean',
        'Espresso',
        'Filter',
        'Plunger',
        'Stovetop',
        'Aeropress',
    );

    /**
     * COGS per size (base multiplier from 250g price)
     */
    private $size_cogs_multiplier = array(
        '250g' => 1.0,
        '1kg' => 3.8,
    );

    /**
     * Import product with customization options
     */
    public function import_product($myshop_product_id, $customization = array()) {
        $myshop_product = yourbeans()->api->get_product($myshop_product_id);

        if (!$myshop_product) {
            return new WP_Error('not_found', 'Product not found in catalog');
        }

        // Create product with CUSTOMIZABLE fields
        $product = new WC_Product_Variable();

        // CUSTOMIZABLE: Title (default to MyShop title if not provided)
        $product->set_name($customization['title'] ?? $myshop_product['title']);

        // CUSTOMIZABLE: Description
        $product->set_description($customization['description'] ?? $myshop_product['description']);
        $product->set_short_description($customization['short_description'] ?? $myshop_product['short_description']);

        $product->set_status('draft'); // Start as draft so owner can customize
        $product->set_catalog_visibility('visible');

        // Set FIXED attributes (cannot be changed)
        $this->set_fixed_attributes($product);

        $product_id = $product->save();

        // Store FIXED MyShop reference (cannot be changed)
        update_post_meta($product_id, '_yourbeans_myshop_product_id', $myshop_product_id);
        update_post_meta($product_id, '_yourbeans_imported_at', current_time('mysql'));

        // Store COGS data (from MyShop, not editable by store owner)
        update_post_meta($product_id, '_yourbeans_cogs_250g', $myshop_product['cogs_250g']);
        update_post_meta($product_id, '_yourbeans_cogs_1kg', $myshop_product['cogs_1kg']);

        // Create FIXED variants (12 total)
        $this->create_fixed_variants($product_id, $myshop_product, $customization);

        // Enable subscription by default
        update_post_meta($product_id, '_yourbeans_subscription_enabled', 'yes');
        update_post_meta($product_id, '_yourbeans_subscription_discount', 10);

        // CUSTOMIZABLE: Images (import defaults, owner can replace)
        if (!empty($myshop_product['images']) && empty($customization['skip_images'])) {
            $this->import_default_images($product_id, $myshop_product['images']);
        }

        return $product_id;
    }

    /**
     * Set FIXED product attributes (grind and size)
     * Store owners cannot modify these
     */
    private function set_fixed_attributes($product) {
        $attributes = array();

        // Size - FIXED
        $size_attr = new WC_Product_Attribute();
        $size_attr->set_id(0);
        $size_attr->set_name('Size');
        $size_attr->set_options($this->fixed_sizes);
        $size_attr->set_position(0);
        $size_attr->set_visible(true);
        $size_attr->set_variation(true);
        $attributes[] = $size_attr;

        // Grind - FIXED
        $grind_attr = new WC_Product_Attribute();
        $grind_attr->set_id(0);
        $grind_attr->set_name('Grind');
        $grind_attr->set_options($this->fixed_grinds);
        $grind_attr->set_position(1);
        $grind_attr->set_visible(true);
        $grind_attr->set_variation(true);
        $attributes[] = $grind_attr;

        $product->set_attributes($attributes);
    }

    /**
     * Create FIXED 12 variants
     * Prices are customizable via markup, but variant structure is fixed
     */
    private function create_fixed_variants($product_id, $myshop_product, $customization) {
        $markup = floatval($customization['markup'] ?? get_option('yourbeans_default_markup', 50)) / 100;

        foreach ($this->fixed_sizes as $size) {
            $cogs_key = 'cogs_' . str_replace('g', '', strtolower($size));
            $cogs = floatval($myshop_product[$cogs_key] ?? $myshop_product['cogs_250g'] * $this->size_cogs_multiplier[$size]);

            foreach ($this->fixed_grinds as $grind) {
                $variation = new WC_Product_Variation();
                $variation->set_parent_id($product_id);

                // CUSTOMIZABLE: Price (via markup)
                $retail_price = $cogs * (1 + $markup);
                $variation->set_regular_price(number_format($retail_price, 2, '.', ''));

                // FIXED: Attributes
                $variation->set_attributes(array(
                    'size' => $size,
                    'grind' => sanitize_title($grind),
                ));

                // FIXED: SKU structure (required for fulfillment)
                $grind_code = strtoupper(substr(str_replace(' ', '', $grind), 0, 4));
                $size_code = str_replace('g', '', $size);
                $sku = sprintf('YB-%d-%s-%s', $myshop_product['id'], $size_code, $grind_code);
                $variation->set_sku($sku);

                // Stock
                $variation->set_manage_stock(true);
                $variation->set_stock_quantity(100);
                $variation->set_stock_status('instock');

                $variation_id = $variation->save();

                // FIXED: COGS (cannot be changed by store owner)
                update_post_meta($variation_id, '_yourbeans_cogs', $cogs);
                update_post_meta($variation_id, '_yourbeans_size', $size);
                update_post_meta($variation_id, '_yourbeans_grind', $grind);
            }
        }

        wc_delete_product_transients($product_id);
    }

    /**
     * Update customizable fields only
     * Called when store owner edits product
     */
    public function update_customization($product_id, $data) {
        $product = wc_get_product($product_id);

        if (!$product || !$this->is_yourbeans_product($product_id)) {
            return new WP_Error('invalid', 'Not a YourBeans product');
        }

        // ALLOWED: Update title
        if (isset($data['title'])) {
            $product->set_name(sanitize_text_field($data['title']));
        }

        // ALLOWED: Update descriptions
        if (isset($data['description'])) {
            $product->set_description(wp_kses_post($data['description']));
        }

        if (isset($data['short_description'])) {
            $product->set_short_description(wp_kses_post($data['short_description']));
        }

        // ALLOWED: Update markup/prices
        if (isset($data['markup'])) {
            $this->update_variant_prices($product_id, floatval($data['markup']));
        }

        // ALLOWED: Update subscription settings
        if (isset($data['subscription_enabled'])) {
            update_post_meta($product_id, '_yourbeans_subscription_enabled', $data['subscription_enabled'] ? 'yes' : 'no');
        }

        if (isset($data['subscription_discount'])) {
            update_post_meta($product_id, '_yourbeans_subscription_discount', absint($data['subscription_discount']));
        }

        $product->save();

        return true;
    }

    /**
     * Prevent modification of fixed attributes
     * Hook into WooCommerce product save
     */
    public function protect_fixed_attributes($product_id) {
        if (!$this->is_yourbeans_product($product_id)) {
            return;
        }

        $product = wc_get_product($product_id);

        // Restore fixed attributes if they were modified
        $this->set_fixed_attributes($product);
        $product->save();
    }
}
```

### 4. Shipping Cost Calculation (`class-yourbeans-shipping.php`)

```php
<?php
class YourBeans_Shipping {

    /**
     * Shipping zones and rates from MyShop
     * These are fetched from MyShop API and cached
     */
    private $shipping_zones = array();

    public function __construct() {
        $this->load_shipping_rates();
    }

    /**
     * Load shipping rates from MyShop or cache
     */
    private function load_shipping_rates() {
        $cached = get_transient('yourbeans_shipping_rates');

        if ($cached) {
            $this->shipping_zones = $cached;
            return;
        }

        try {
            $rates = yourbeans()->api->get_shipping_rates();
            $this->shipping_zones = $rates;
            set_transient('yourbeans_shipping_rates', $rates, DAY_IN_SECONDS);
        } catch (Exception $e) {
            // Fallback to default rates
            $this->shipping_zones = $this->get_default_rates();
        }
    }

    /**
     * Calculate shipping cost for order
     */
    public function calculate_shipping_cost($order) {
        $shipping_address = $order->get_shipping_state() ?: $order->get_billing_state();
        $shipping_postcode = $order->get_shipping_postcode() ?: $order->get_billing_postcode();

        $zone = $this->determine_zone($shipping_address, $shipping_postcode);
        $total_weight = $this->calculate_order_weight($order);

        return $this->get_zone_rate($zone, $total_weight);
    }

    /**
     * Calculate total weight of YourBeans items
     */
    private function calculate_order_weight($order) {
        $total_weight_grams = 0;

        foreach ($order->get_items() as $item) {
            $product_id = $item->get_product_id();

            if (!get_post_meta($product_id, '_yourbeans_myshop_product_id', true)) {
                continue; // Skip non-YourBeans items
            }

            $variation_id = $item->get_variation_id();
            $size = get_post_meta($variation_id, '_yourbeans_size', true);
            $quantity = $item->get_quantity();

            $weight = ($size === '1kg') ? 1000 : 250;
            $total_weight_grams += ($weight * $quantity);
        }

        return $total_weight_grams / 1000; // Return in kg
    }

    /**
     * Determine shipping zone from address
     */
    private function determine_zone($state, $postcode) {
        $metro_states = array('NSW', 'VIC', 'QLD', 'SA', 'ACT');
        $remote_states = array('WA', 'NT', 'TAS');

        // Check postcode for regional/remote within metro states
        $metro_postcodes = array('2000-2249', '3000-3207', '4000-4179', '5000-5199', '2600-2618');

        if (in_array($state, $remote_states)) {
            return 'remote';
        }

        if (in_array($state, $metro_states)) {
            // Check if postcode is metro
            if ($this->is_metro_postcode($postcode, $state)) {
                return 'metro';
            }
            return 'regional';
        }

        return 'regional'; // Default
    }

    /**
     * Get shipping rate for zone and weight
     */
    private function get_zone_rate($zone, $weight_kg) {
        $rates = $this->shipping_zones[$zone] ?? $this->shipping_zones['regional'];

        $base_rate = $rates['base'] ?? 8.50;
        $per_kg_rate = $rates['per_kg'] ?? 2.00;

        // First kg included in base rate
        $extra_kg = max(0, ceil($weight_kg) - 1);

        return $base_rate + ($extra_kg * $per_kg_rate);
    }

    /**
     * Default shipping rates (fallback)
     */
    private function get_default_rates() {
        return array(
            'metro' => array(
                'base' => 8.50,
                'per_kg' => 2.00,
            ),
            'regional' => array(
                'base' => 10.50,
                'per_kg' => 2.50,
            ),
            'remote' => array(
                'base' => 14.50,
                'per_kg' => 3.00,
            ),
        );
    }
}
```

---

## Admin Interface: COGS & Settings

### COGS Display in Admin

Store owners can VIEW (not edit) COGS in the admin:

```php
// In admin product edit page
add_action('woocommerce_product_options_pricing', 'yourbeans_display_cogs_info');

function yourbeans_display_cogs_info() {
    global $post;

    if (!get_post_meta($post->ID, '_yourbeans_myshop_product_id', true)) {
        return;
    }

    $cogs_250g = get_post_meta($post->ID, '_yourbeans_cogs_250g', true);
    $cogs_1kg = get_post_meta($post->ID, '_yourbeans_cogs_1kg', true);
    ?>
    <div class="yourbeans-cogs-info" style="background:#f0f0f0; padding:10px; margin:10px 0;">
        <h4 style="margin:0 0 10px;">YourBeans Cost Information</h4>
        <p><strong>250g COGS:</strong> $<?php echo esc_html(number_format($cogs_250g, 2)); ?></p>
        <p><strong>1kg COGS:</strong> $<?php echo esc_html(number_format($cogs_1kg, 2)); ?></p>
        <p class="description">
            Cost of Goods Sold is set by MyShop.Coffee and cannot be modified.
            Your margin = Retail Price - COGS - Shipping
        </p>
    </div>
    <?php
}
```

### Settings Page Structure

```php
// admin/views/settings.php sections:

1. LICENSE STATUS
   - Subscription status (Active/Inactive)
   - Manage Subscription button (links to Stripe Portal)
   - License key field

2. STRIPE CONNECT
   - Connection status
   - Connect/Disconnect button
   - Connected account ID

3. DEFAULT SETTINGS
   - Default markup % (applies to new imports)
   - Auto-sync inventory (on/off)
   - Default subscription discount %

4. SHIPPING ZONES (Read-only, from MyShop)
   - Metro rates
   - Regional rates
   - Remote rates

5. SUBSCRIPTION SETTINGS
   - Enable subscriptions by default
   - Available intervals
   - Default discount percentage
```

---

## API Endpoints Required (MyShop.Coffee Side)

### License Management

| Endpoint | Method | Description |
|----------|--------|-------------|
| `/yourbeans/v1/license/verify` | POST | Verify license key and subscription status |
| `/yourbeans/v1/license/billing-portal` | POST | Get Stripe billing portal URL |
| `/yourbeans/v1/license/activate` | POST | Activate license for a store |

### Products

| Endpoint | Method | Description |
|----------|--------|-------------|
| `/yourbeans/v1/products` | GET | List available products with COGS |
| `/yourbeans/v1/products/{id}` | GET | Get product with COGS per variant |
| `/yourbeans/v1/shipping-rates` | GET | Get current shipping rates by zone |

### Orders

| Endpoint | Method | Description |
|----------|--------|-------------|
| `/yourbeans/v1/orders` | POST | Submit order for fulfillment |
| `/yourbeans/v1/orders/{id}/status` | GET | Get fulfillment status |

---

## Data Flow Summary

```
┌─────────────────────────────────────────────────────────────────┐
│                    STORE OWNER JOURNEY                          │
└─────────────────────────────────────────────────────────────────┘

1. SUBSCRIBE TO PLUGIN
   └── Pay via Stripe → MyShop Enterprises
   └── Receive license key
   └── Install & activate plugin

2. CONNECT STRIPE
   └── OAuth flow → Stripe Connect
   └── Enable payment splitting

3. IMPORT PRODUCTS
   └── Browse MyShop catalog
   └── Click Import → Creates WC product
   └── 12 variants auto-created (FIXED)
   └── COGS stored per variant (FIXED)

4. CUSTOMIZE BRANDING
   └── Edit title, description, images (ALLOWED)
   └── Set markup/pricing (ALLOWED)
   └── Cannot change: grinds, sizes, SKUs (FIXED)

5. SELL TO CUSTOMERS
   └── One-time purchases
   └── Subscription purchases (weekly/fortnightly/monthly)

6. ORDER FULFILLMENT
   └── Payment split: COGS + Shipping → MyShop
   └── Order sent to MyShop API
   └── MyShop fulfills & ships
   └── Tracking pushed back to store
```

---

## Implementation Phases

### Phase 1: Core + Licensing
- [ ] Plugin structure
- [ ] License verification with MyShop API
- [ ] Basic settings page
- [ ] Stripe billing portal integration

### Phase 2: Products + Customization
- [ ] Product import with fixed variants
- [ ] COGS storage per variant
- [ ] Customization rules (title, description, images)
- [ ] Protected attributes (grind, size)

### Phase 3: Payment Splitting
- [ ] Stripe Connect OAuth
- [ ] COGS + Shipping calculation
- [ ] Platform fee in payment intent
- [ ] Order submission to MyShop

### Phase 4: Subscriptions
- [ ] Subscription options on product page
- [ ] Cart/checkout subscription handling
- [ ] Renewal scheduling (cron)
- [ ] Customer subscription management

### Phase 5: Polish
- [ ] Inventory sync
- [ ] Email notifications
- [ ] Error handling
- [ ] Documentation
